#pragma once

#include <vector>
#include <string>
#include <openplx/NodeType.h>
#include <openplx/Token.h>
#include <openplx/RefactorOps.h>

namespace openplx::Refactor {

    /**
     * Utility methods for refactoring a .openplx code base. The methods takes a list of documents and
     * some parameters for the specific refactoring operation and returns a list of "operations" that
     * encode how to update the .openplx files to realize the refactorization.
     */
    class Toolkit {
        public:
            /**
             * Renames an existing model given by `old_name_with_namespace`,
             * the new name given by `new_name` should not include a namespace.
             * To move a model to another namespace use `moveModel`.
             *
             * Example:
             * `renameModel(documents, "Physics3D.Interactions.CoolMotor", "SweetMotor")`
             */
            static std::vector<ReplaceOp> renameModel(
                std::vector<DocPtr> documents,
                std::string old_name_with_namespace,
                std::string new_name);

            /**
             * Moves an existing model given by `model_name_with_namespace`
             * to a new namespace given by `new_namespace`,
             *
             * Example:
             * `moveModel(documents, "Physics3D.Interactions.CoolMotor", "Physics3D.Motors")`
             */
            static std::vector<ReplaceOp> moveModel(
                std::vector<DocPtr> documents,
                std::string model_name_with_namespace,
                std::string new_namespace
            );

            /**
             * Moves an existing model given by `model_name_with_namespace`
             * to a new namespace given by `new_namespace`, and changes it name to `new_name`
             *
             * Example:
             * `moveAndRenameModel(documents, "Physics3D.Transform", "Math", "AffineTransform")`
             */
            static std::vector<ReplaceOp> moveAndRenameModel(
                std::vector<DocPtr> documents,
                std::string model_name_with_namespace,
                std::string new_namespace,
                std::string new_name
            );

            /**
             * Rename a namespace.
             *
             * Example:
             * `renameNamespace(documents, "Physics3D.Interactions", "Physics3D.Mates")`
             */
            static std::vector<ReplaceOp> renameNamespace(
                std::vector<DocPtr> documents,
                std::string old_namespace,
                std::string new_namespace);

            /**
             * Rename an attribute in a specific model given by `old_name_with_path`
             *
             * Example:
             * `renameAttribute(documents, "Physics3D.Interactions.CoolMotor.power", "gain")`
             */
            static std::vector<ReplaceOp> renameAttribute(
                std::vector<DocPtr> documents,
                std::string old_name_with_path,
                std::string new_name);

            /**
             * Rename an method in a specific model given by `old_name_with_path`
             *
             * Example:
             * `renameAttribute(documents, "Physics3D.Interactions.CoolMotor.turn_on", "engage")`
             */
            static std::vector<ReplaceOp> renameMethod(
                std::vector<DocPtr> documents,
                std::string old_name_with_path,
                std::string new_name);
    };
}
