
#pragma once

#include <memory>
#include <vector>
#include <unordered_set>
#include <string>
#include <openplx/Token.h>
#include <openplx/NodeType.h>
#include <openplx/NodeVisitor.h>

namespace openplx
{
    class NodeVisitor;

    class Node
    {
        public:
            NodeType getNodeType() const;
            bool isValid() const;
            void setValid(bool valid);
            std::string getId() const;
            void setId(std::string id);

            bool isDocument() const;
            bool isModelDeclaration() const;
            bool isVarDeclaration() const;
            bool isVarAssignment() const;
            bool isMethodDeclaration() const;
            bool isParameter() const;
            bool isConstant() const;
            bool isBinaryOp() const;
            bool isArray() const;
            bool isArrayType() const;
            bool isUnary() const;
            bool isMemberAccess() const;
            bool isCall() const;
            bool isPrimitiveType() const;
            bool isType() const;
            bool isCompositeType() const;
            bool isIndexing() const;
            bool isOperatorOverload() const;
            bool isAnnotation() const;
            bool isImport() const;
            bool isTraitImpl() const;
            bool isInitializer() const;
            bool isInitializerType() const;
            bool isDeletion() const;
            bool isWithType() const;
            bool isOrType() const;
            bool isInnerSymbol() const;
            bool isOuterSymbol() const;

            virtual DocPtr asDocument();
            virtual ModelDeclPtr asModelDeclaration();
            virtual VarDeclPtr asVarDeclaration();
            virtual VarAssignPtr asVarAssignment();
            virtual MethodDeclPtr asMethodDeclaration();
            virtual ParamPtr asParameter();
            virtual ConstantPtr asConstant();
            virtual BinaryOpPtr asBinaryOp();
            virtual UnaryPtr asUnary();
            virtual ArrayPtr asArray();
            virtual ArrayTypePtr asArrayType();
            virtual MemberAccessPtr asMemberAccess();
            virtual CallPtr asCall();
            virtual PrimitiveTypePtr asPrimitiveType();
            virtual TypePtr asType();
            virtual CompositeTypePtr asCompositeType();
            virtual IndexingPtr asIndexing();
            virtual OpOverloadPtr asOperatorOverload();
            virtual AnnotationPtr asAnnotation();
            virtual ImportPtr asImport();
            virtual TraitImplPtr asTraitImpl();
            virtual InitializerPtr asInitializer();
            virtual InitializerTypePtr asInitializerType();
            virtual DeletionPtr asDeletion();
            virtual WithTypePtr asWithType();
            virtual OrTypePtr asOrType();
            virtual InnerSymbolPtr asInnerSymbol();
            virtual OuterSymbolPtr asOuterSymbol();

            virtual void accept(NodeVisitor& visitor) = 0;

            virtual std::string toString();

            virtual void unbind();
            virtual ~Node() = default;

            static std::string segmentsAsString(const std::vector<Token>& segments);
            static std::string segmentsAsStringSkipLast(const std::vector<Token>& segments);
            static std::string segmentsUpToAsString(const std::vector<Token> &segments, size_t index);

            /** Nodes needs to be repeatedly sorted using Kahn's algorithm which
             *  requires each node to have a degree and an adjacency list.
             **/
            void clearKahn();
            size_t getDegree() const;
            void incrDegree();
            size_t decrDegree();
            const std::unordered_set<NodePtr>& getEdges() const;
            bool appendToEdges(NodePtr node);
            bool appendToEdgesIfInSet(NodePtr node, const std::unordered_set<NodePtr>& node_set);

        protected:
            Node(NodeType node_type);

        private:
            NodeType m_node_type;
            bool m_valid;
            std::string m_id;
            std::unordered_set<NodePtr> m_edges;
            size_t m_degree;
    };
}
