#pragma once

#include <string>
#include <vector>
#include <memory>
#include <utility>
#include <openplx/Types.h>
#include <openplx/Token.h>
#include <openplx/ParserError.h>
#include <openplx/kernel_export.h>


#ifdef _MSC_VER
#  pragma warning(push)
#  pragma warning(disable: 4251)
#endif

namespace openplx
{

class OPENPLX_KERNEL_EXPORT Error
{
public:
    template <class T, class ...Args>
    static std::shared_ptr<T> createOnToken( const Token& token, const std::string& source_id, Args&& ...args);

    template <class T, class ...Args>
    static std::shared_ptr<T> createOnTokens( const std::vector<Token>& token, const std::string& source_id, Args&& ...args);

    template <class T, class ...Args>
    static std::shared_ptr<T> create( line_t line, line_t column, const std::string& source_id, Args&& ...args);

    template <class T, class ...Args>
    static std::shared_ptr<T> create( line_t from_line, col_t from_column, line_t to_line, line_t to_column, const std::string& source_id, Args&& ...args);

    error_code_t getErrorCode() const;
    line_t getLine() const;
    col_t getColumn() const;
    const std::string& getSourceId() const;

    std::string getMessage(bool decorate = false) const;

    line_t getFromLine() const;
    col_t getFromColumn() const;
    line_t getToLine() const;
    col_t getToColumn() const;

    virtual ~Error() = default;

    template<typename T>
    static std::shared_ptr<T> as(std::shared_ptr<Error> error) {
        return std::dynamic_pointer_cast<T>(error);
    }

protected:
    Error(
        error_code_t error_code,
        line_t from_line,
        col_t from_column,
        line_t to_line,
        line_t to_column,
        const std::string& source_id);

        virtual std::string createErrorMessage() const;
private:
    error_code_t m_error_code;
    line_t m_from_line;
    col_t m_from_column;
    line_t m_to_line;
    col_t m_to_column;
    std::string m_source_id;
};

template <class T, class ...Args>
std::shared_ptr<T> Error::createOnToken( const Token& token, const std::string& source_id, Args&& ...args){
    auto end_of = Token::end_of(token);
    return std::shared_ptr<T>(new T(token.line,token.column, end_of.first, end_of.second, source_id, std::forward<Args>(args)...));
}

template <class T, class ...Args>
std::shared_ptr<T> Error::createOnTokens( const std::vector<Token>& tokens, const std::string& source_id, Args&& ...args){
    if (tokens.empty())
        return std::shared_ptr<T>(new T(1, 1, 1, 1, source_id, args...));
    auto end_of = Token::end_of(tokens.back());
    return std::shared_ptr<T>(new T(tokens[0].line, tokens[0].column, end_of.first, end_of.second, source_id, std::forward<Args>(args)...));
}

template <class T, class ...Args>
std::shared_ptr<T> Error::create( line_t line, col_t column, const std::string& source_id, Args&& ...args){
    return std::shared_ptr<T>(new T(line, column, line, column, source_id, std::forward<Args>(args)...));
}

template <class T, class ...Args>
std::shared_ptr<T> Error::create( line_t from_line, col_t from_column, line_t to_line, col_t to_column, const std::string& source_id, Args&& ...args){
    return std::shared_ptr<T>(new T(from_line, from_column, to_line, to_column, source_id, std::forward<Args>(args)...));
}

// Basic Errors
#define OPENPLX_BASIC_ERROR(name, code)                                                                         \
class OPENPLX_KERNEL_EXPORT name : public Error {                                                                 \
public:                                                                                                         \
    static constexpr error_code_t ErrorCode = code;                                                             \
    name(line_t from_line, col_t from_column, line_t to_line, col_t to_column, const std::string& source_id);   \
protected:                                                                                                      \
    virtual std::string createErrorMessage() const override;                                                    \
};
// Syntax Errors
OPENPLX_BASIC_ERROR(ExpectedDeclarationName, SyntaxError::ExpectedDeclarationName)
OPENPLX_BASIC_ERROR(ExpectedMembers, SyntaxError::ExpectedMembers)
OPENPLX_BASIC_ERROR(ExpectedIndentedBlock, SyntaxError::ExpectedIndentedBlock)
OPENPLX_BASIC_ERROR(UnexpectedCharacter, SyntaxError::UnexpectedCharacter)
OPENPLX_BASIC_ERROR(ExpectedIdentifier, SyntaxError::ExpectedIdentifier)
OPENPLX_BASIC_ERROR(UnexpectedIndentation, SyntaxError::UnexpectedIndentation)
OPENPLX_BASIC_ERROR(UnexpectedToken, SyntaxError::UnexpectedToken)
OPENPLX_BASIC_ERROR(UnexpectedEndOfLine, SyntaxError::UnexpectedEndOfLine)
OPENPLX_BASIC_ERROR(ExpectedReturnType, SyntaxError::ExpectedReturnType)
OPENPLX_BASIC_ERROR(ExpectedSymbol, SyntaxError::ExpectedSymbol)
OPENPLX_BASIC_ERROR(ExpectedValue, SyntaxError::ExpectedValue)
OPENPLX_BASIC_ERROR(ExpectedIs, SyntaxError::ExpectedIs)
OPENPLX_BASIC_ERROR(ExpectedType, SyntaxError::ExpectedType)
OPENPLX_BASIC_ERROR(UnexpectedEndOfFile, SyntaxError::UnexpectedEndOfFile)
OPENPLX_BASIC_ERROR(ExpectedColon, SyntaxError::ExpectedColon)
OPENPLX_BASIC_ERROR(InvalidExpression, SyntaxError::InvalidExpression)
OPENPLX_BASIC_ERROR(UnterminatedString, SyntaxError::UnterminatedString)
OPENPLX_BASIC_ERROR(ExpectedOperator, SyntaxError::ExpectedOperator)
OPENPLX_BASIC_ERROR(TooComplexAnnotationExpression, SyntaxError::TooComplexAnnotationExpression)
OPENPLX_BASIC_ERROR(InvalidModelQualifier, SyntaxError::InvalidModelQualifier)
OPENPLX_BASIC_ERROR(InvalidIdentifierString, SyntaxError::InvalidIdentifierString)
OPENPLX_BASIC_ERROR(ComplexTypeExpressionsNotAllowed, SyntaxError::ComplexTypeExpressionsNotAllowed)
OPENPLX_BASIC_ERROR(InvalidDollarString, SyntaxError::InvalidDollarString)
// Semantic Errors
OPENPLX_BASIC_ERROR(WrongNumberOfArguments, SemanticError::WrongNumberOfArguments)
OPENPLX_BASIC_ERROR(InvalidNumberOfParameters, SemanticError::InvalidNumberOfParameters)
OPENPLX_BASIC_ERROR(OperatorAlreadyDefined, SemanticError::OperatorAlreadyDefined)
OPENPLX_BASIC_ERROR(AssignedVarInConstModel, SemanticError::AssignedVarInConstModel)
OPENPLX_BASIC_ERROR(MissingConstWhenParentConst, SemanticError::MissingConstWhenParentConst)
OPENPLX_BASIC_ERROR(NestedDeclarationsNotAllowed, SemanticError::NestedDeclarationsNotAllowed)
OPENPLX_BASIC_ERROR(NestedTraitsNotAllowed, SemanticError::NestedTraitsNotAllowed)
OPENPLX_BASIC_ERROR(TraitsWithCommonAncestorNotAllowed, SemanticError::TraitsWithCommonAncestorNotAllowed)
OPENPLX_BASIC_ERROR(AssignedVarInTrait, SemanticError::AssignedVarInTrait)
OPENPLX_BASIC_ERROR(MissingSpecialization, SemanticError::MissingSpecialization)
OPENPLX_BASIC_ERROR(OptionalCannotBecome, SemanticError::OptionalCannotBecome)
OPENPLX_BASIC_ERROR(RhsMustBeReference, SemanticError::RhsMustBeReference)
// Runtime Errors
OPENPLX_BASIC_ERROR(DivisionByZero, RuntimeError::DivisionByZero)
OPENPLX_BASIC_ERROR(NotInstantiated, RuntimeError::NotInstantiated)
OPENPLX_BASIC_ERROR(IndexOutOfBounds, RuntimeError::IndexOutOfBounds)
OPENPLX_BASIC_ERROR(MissingImplementation, RuntimeError::MissingImplementation)
OPENPLX_BASIC_ERROR(IntOutOfBounds, RuntimeError::IntOutOfBounds)
#undef OPENPLX_BASIC_ERRORS

class OPENPLX_KERNEL_EXPORT CycleFoundError : public Error
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::CycleFound;

    std::string getCyclicNodesAsString() const;

    std::string getModelName() const;
    static std::shared_ptr<CycleFoundError> create(const Token& token,
                                                    const std::string& source_id,
                                                    const std::string& model_name, const std::string& cyclic_nodes);

    CycleFoundError(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& model_name,
        const std::string& cyclic_nodes);

protected:
    virtual std::string createErrorMessage() const override;

private:
    std::string m_model_name;
    std::string m_cyclic_nodes;
};



// A generic Error that has a String parameter
class OPENPLX_KERNEL_EXPORT StringParameterError : public Error
{
public:
    std::string getStringParameter() const;
    static std::shared_ptr<StringParameterError> create(error_code_t error_code,
                                                        line_t line, col_t column,
                                                            const std::string& source_id,
                                                        const std::string& string_parameter);

    StringParameterError(
        error_code_t error_code,
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& string_parameter);

protected:
    virtual std::string createErrorMessage() const override;

private:
    std::string m_string_parameter;
};



class OPENPLX_KERNEL_EXPORT MissingAssignmentError : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::MissingAssignment;

    static std::shared_ptr<MissingAssignmentError> create(line_t line, col_t column,
                                                            const std::string& source_id,
                                                            const std::string& var_decl_path);
    std::string getVarDeclarationPath() const;

    MissingAssignmentError(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& var_decl_path);

protected:
    virtual std::string createErrorMessage() const override;
};



class OPENPLX_KERNEL_EXPORT CircularDependency : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::CircularDependency;

    std::string getCycleAsString() const;
    static std::shared_ptr<CircularDependency> create(line_t line, col_t column,
                                                        const std::string& source_id,
                                                        const std::string& cyclic_nodes);

    CircularDependency(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& cyclic_nodes);

protected:
    virtual std::string createErrorMessage() const override;
};



class OPENPLX_KERNEL_EXPORT ModelDeclarationNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::ModelDeclarationNotFound;

    std::string getModelName() const;
    static std::shared_ptr<ModelDeclarationNotFound> create(line_t line, col_t column,
                                                            const std::string& source_id,
                                                            const std::string& model_name);

    ModelDeclarationNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& model_name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT TypeNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::TypeNotFound;

    std::string getTypeSegments() const;
    static std::shared_ptr<TypeNotFound> create(const std::vector<Token>& type_segments, const std::string& source_id);

    TypeNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& type_segments);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT VariableNotFound : public StringParameterError
{
public:
    static std::shared_ptr<VariableNotFound> create(line_t line, col_t column,
                                                    const std::string& source_id,
                                                    const std::string& name);

    VariableNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT MemberNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::MemberNotFound;

    static std::shared_ptr<MemberNotFound> create(const Token& member_token, const std::string& source_id);

    MemberNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& member_identifier);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT MethodNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::MethodNotFound;

    static std::shared_ptr<MethodNotFound> create(const Token& name_token, const std::string& method_path, const std::string& source_id);

    MethodNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& method_path);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT SymbolAlreadyDefined : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::SymbolAlreadyDefined;

    static std::shared_ptr<SymbolAlreadyDefined> create(line_t line, col_t column,
                                                    const std::string &source_id,
                                                    const std::string &name);

    SymbolAlreadyDefined(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT SymbolAlreadyAssigned : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::SymbolAlreadyAssigned;

    static std::shared_ptr<SymbolAlreadyAssigned> create(const Token& identifier,
                                                    const std::string& source_id);

    SymbolAlreadyAssigned(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ImportedFileNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SyntaxError::ImportedFileNotFound;

    static std::shared_ptr<ImportedFileNotFound> create(line_t line, col_t column,
                                                    const std::string& source_id,
                                                    const std::string& name);

    ImportedFileNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT NoPluginForImportedFile : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SyntaxError::NoPluginForImportedFile;

    static std::shared_ptr<NoPluginForImportedFile> create(line_t line, col_t column,
                                                            const std::string& source_id,
                                                            const std::string& name);

    NoPluginForImportedFile(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT FileNotFound : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::FileNotFound;

    static std::shared_ptr<FileNotFound> create(line_t line, col_t column,
                                                const std::string &source_id,
                                                const std::string &name);

    FileNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ExpectedTrait : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::ExpectedTrait;

    static std::shared_ptr<ExpectedTrait> create(const std::vector<Token>& segments,
                                                    const std::string& source_id);

    ExpectedTrait(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& trait);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ModelNotInitializable : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::ModelNotInitializable;

    static std::shared_ptr<ModelNotInitializable> create(
        line_t line, col_t column,
        const std::string& model_name,
        const std::string& source_id);

    ModelNotInitializable(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& model_name);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT DependencyNotSatisfied : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::RuntimeError::DependencyNotSatisfied;

    static std::shared_ptr<DependencyNotSatisfied> create(
        line_t line, col_t column,
        const std::string& dependency,
        const std::string& source_id);

    DependencyNotSatisfied(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& dependency,
        const std::string& source_id);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ReferencesAreReadOnly : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::ReferencesAreReadOnly;

    static std::shared_ptr<ReferencesAreReadOnly> create(
        line_t line, col_t column,
        const std::string& reference,
        const std::string& source_id);

    ReferencesAreReadOnly(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& reference,
        const std::string& source_id);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ReferencesMustBeModels : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::ReferencesMustBeModels;

    static std::shared_ptr<ReferencesMustBeModels> create(
        Token first_token, Token reference_token,
        const std::string& reference,
        const std::string& source_id);

    ReferencesMustBeModels(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& reference,
        const std::string& source_id);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT ReferencesCannotBecome : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::ReferencesCannotBecome;

    static std::shared_ptr<ReferencesCannotBecome> create(
        Token first_token, Token reference_token,
        const std::string& reference,
        const std::string& source_id);

    ReferencesCannotBecome(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& reference,
        const std::string& source_id);

protected:
    virtual std::string createErrorMessage() const override;
};


class OPENPLX_KERNEL_EXPORT TypeMismatch : public StringParameterError
{
public:
    static constexpr error_code_t ErrorCode = openplx::SemanticError::TypeMismatch;

    static std::shared_ptr<TypeMismatch> create(
        Token first_token, Token last_token,
        const std::string& lhs_type,
        const std::string& rhs_type,
        const std::string& source_id);

    TypeMismatch(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& string_param,
        const std::string& source_id);

protected:
    virtual std::string createErrorMessage() const override;
};

class OPENPLX_KERNEL_EXPORT InvalidDocumentSeparator : public StringParameterError
{
    public:
        static constexpr error_code_t ErrorCode = openplx::SemanticError::InvalidDocumentSeparator;

        static std::shared_ptr<InvalidDocumentSeparator> create(Token first_token, Token last_token,
                                                                const std::string &source_id,
                                                                const std::string &message);

        InvalidDocumentSeparator(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& source_id,
            const std::string& message);

    protected:
        virtual std::string createErrorMessage() const override;
};


using Errors = std::vector<std::shared_ptr<Error>>;
}

#ifdef _MSC_VER
#  pragma warning(pop)
#endif
