#pragma once

#include <openplx/control_export.h>
#include <memory>
#include <string>
#include <cstdint>
#include <unordered_set>
#include <unordered_map>
#include <optional>

namespace openplx
{
    namespace Core {
        class Object;
    }

    namespace Physics::Signals {
        class Output;
        class Input;
    }

    class Marshalling;
    class DetachedControlImpl;

    enum class VirtualControlType {
        NotVirtual,
        BroadcastInput,
        AffineTransform1DInput,
        AffineTransform1DOutput,
        GroupOutput
    };

    class OPENPLX_CONTROL_EXPORT DetachedControl
    {
        public:
            DetachedControl();
            ~DetachedControl();

            bool is_input() const;
            bool is_output() const;
            uint32_t get_key() const;
            const std::string& get_name() const;
            const std::string& get_uuid() const;
            const std::string& get_type() const;
            const std::shared_ptr<Marshalling>& get_marshalling() const;
            const std::shared_ptr<Core::Object>& get_attached_reference() const;

            bool is_virtual() const;
            VirtualControlType get_virtual_control_type() const;
            const std::unordered_set<uint32_t>& get_virtual_dependencies() const;
            size_t get_degree() const;
            void set_degree(size_t degree);
            void add_edge(uint32_t edge);
            const std::unordered_set<uint32_t>& get_edges() const;
            void clear_edges();

            void set_is_input();
            void set_is_output();
            void set_key(uint32_t key);
            void set_name(std::string name);
            void set_uuid(std::string uuid);
            void set_type(std::string type);
            void set_marshalling(std::shared_ptr<Marshalling> marshalling);
            void set_virtual_control_type(VirtualControlType virtual_control_type);
            void add_virtual_dependency(uint32_t key);
            void set_attached_reference(std::shared_ptr<Core::Object> reference);

            template <typename T>
            std::optional<T> lookup_in_key_map(uint32_t key);

            template <typename T>
            void set_in_key_map(uint32_t key, T value);

            template <typename T>
            const std::unordered_map<uint32_t, T>& get_key_map() const;

            void set_from_output(const std::shared_ptr<Physics::Signals::Output>& output);
            void set_from_input(const std::shared_ptr<Physics::Signals::Input>& input);

        private:
            DetachedControlImpl* m_impl;
    };
}
