/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <utility>

#include <agx/agxPhysics_export.h>
#include <agx/AffineMatrix4x4.h>
#include <agx/ref_ptr.h>
#include <agx/stdint.h>
#include <agx/Vector.h>

#include <agxSDK/Collection.h>


namespace agx
{
  class RigidBody;
  class Constraint;
}

namespace agxSDK {
  class Simulation;
}

namespace agxUtil
{

  /**
  Struct that specifies a position for a constraint angle.
  */
  struct AGXPHYSICS_EXPORT ConstraintPosition {

    /// Constraint
    agx::Constraint* constraint;

    /// Constraint angle position.
    agx::Real        position;

    // Angle index, 0 in most cases.
    agx::UInt8       index;

    inline ConstraintPosition(agx::Constraint* _c, agx::Real _p, agx::UInt8 _i = 0) :
      constraint(_c), position(_p), index(_i)
    {}
  };


  using ConstraintPositionVector = agx::Vector<ConstraintPosition>;

  using BodyTransform = std::pair<agx::RigidBody*, agx::AffineMatrix4x4>;
  using BodyTransformVector = agx::Vector<agxUtil::BodyTransform>;



  /**
  This class can be used to compute transforms for a collection of constrained bodies
  given a set of positions for the constraint angles.


  \warning All simulation features in AGX are not supported and there are some limitations.

           Wires are not supported. A change of pose for a mechanical structure
           could lead to the wire coming in contact with geometry, new contact nodes added,
           leading to resolution changes and this does not fit well with the output format for
           the computed results. It is recommended to remove the wire(s), reconfigure the machine and
           then add wires again so they can be re-routed.

           PowerLine related components can be affected by sudden transform changes and constraint positional-
           and rotational values being changed. The main thing to pay attention to are position of hydraulic pistons
           and the possible change to the constraint the piston drives.


  \warning The computed results should not be applied blindly. Depending on the mechanical structure and
           changes to the constraint positions, there can be needed post operations that are needed
           to keep the simulation in a good state:
           - Handling of motors with 0 speed and setLockedAtZero.
           - Winding number for hinges when changing angle multiple times when ranges are also in use.

  */
  class AGXPHYSICS_EXPORT ReconfigureRequest {

    public:

      ReconfigureRequest();

      ~ReconfigureRequest();


      /**
      Compute new transforms for items in the input collection. The computations are performed so that
      the specified referenceBody, which must be part of the collection, is treated as static and will not move.

      Which body that is used for reference affects the result. For example, in a machine with waist steering,
      the result will be different for the same waist angle if the reference body is in the front- or back
      section of the machine.

      How the mechanical structure should change is specified in the positions vector. The elements in the vector
      consists of (constraint, angle_index, angle_value) having the desired position for the joints.
      The angle index is 0 in most cases, for example for hinge and prismatic where there is only one degree of freedom to control.
      For other constraints, such as cylindrical, the index can take a non-zero value.

      The new state is then computed by using copies of the input items. Constraint controllers are used to
      reach the new state.

      \note Similar result as this method computes can be achieved by setting suitable values on constraint
            motors and simulating forward and then reading the actual body transforms.
            This method also uses constraint controllers, but on copies of the input bodies and constraints.
            The advantage here is that just parts of the scene is included giving the benefits that:
            - Faster to simulate just part of the scene
            - No need to "freeze" parts of the simulation that should be unaffected.
            - Current timestamp in the simulation does not change

      \param collection The bodies and constraint making up the structure that should have a new configuration
      \param referenceBody This body will be kept fixed when performing the movements.
      \param positions Vector with constraint positions.
      \param results Vector with (body,transform) pairs.
      \param maxViolation Optional output of largest constraint violation found at new pose.
      \param includeGeometry If geometry should be included in the copies and used in the computations.
      \param additionalSteps User configurable number of extra steps to take when moving non-stiff structures.
             Increases computation time, but improves results for e.g cables.
      */
      bool computeTransforms(const agxSDK::Collection* collection,
                             const agx::RigidBody* referenceBody,
                             const agxUtil::ConstraintPositionVector& positions,
                             agxUtil::BodyTransformVector& results,
                             agx::Real* maxViolation = nullptr,
                             bool includeGeometry = false,
                             size_t additionalSteps = 0);



      /**
      This method applies the results from computeTransforms by updating body transforms.
      To maintain a stable simulation and reduce risk of artifacts after the instant modification of transforms,
      additional steps are also performed.
      - Hinges which had new joint positions requested in the input positions vector will have their winding number checked and corrected if needed.
      - Constraints in the input positions vector that had Lock controllers enabled will have them disabled and instead motor with 0 speed enabled.
      - All constraints part of the input collection will be checked to avoid an old position being used by a motor controllers if setLockedAtZero is enabled.

      \warning This method can perform additional steps to help with maintaining a good state after updating the body transforms,
               but only for features supported by ReconfigureRequest that also are in the agxPhysics shared library.

               Therefore extra handling can be needed by the user even if this method is used. One such example is when hydraulics is used
               with PistonActuators. If the constraint connected to the piston have its position changed when body transforms are updated,
               then there will be a mismatch between the hydraulic cylinder and the body state. An example of how this can be adjusted
               by updating fluid in the piston chambers is available in the python tutorial tutorial_machine_reconfiguration.agxPy.
      */
      void applyTransforms(agxSDK::Collection* collection,
                           const agxUtil::ConstraintPositionVector& positions,
                           agxUtil::BodyTransformVector& transforms,
                           bool zeroVelocities = true);


      /**
      Return textual description of problem if computeTransforms returned false.
      */
      agx::String getErrorMessage() const;

    private:
      agx::ref_ptr<agxSDK::Simulation> m_simulation;

      agx::RigidBodyRefVector          m_bodies;
      agx::ConstraintRefVector         m_constraints;

      agx::UInt64                      m_state;
      agx::String                      m_lastErrorMessage;

  };

}

