/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/BitState.h>
#include <agxTerrain/CompositeShovel.h>
#include <agxTerrain/Shovel.h>
#include <agxStream/Serializable.h>

namespace agx
{
  class RigidBody;
}

namespace agxCollide
{
  class Geometry;
}

namespace agxTerrain
{
  AGX_DECLARE_POINTER_TYPES( ClamShellDebugRenderSettings );

  AGX_DECLARE_POINTER_TYPES( ClamShellBucket );
  AGX_DECLARE_VECTOR_TYPES( ClamShellBucket );

  /**
  ClamShellBucket is a CompositeShovel class that models a clamshell
  bucket that consists of two opposing shovels. This class manages
  shovel functionality when two shovels have overlapping soil aggregates.
  Examples of this is making sure we do not double count particles and
  also to	create contacts between the opposing shovel aggregates when
  the bucket is closed.

  The ClamShellBucket is considered closed when the distance between the cutting
  edges between the two shovels in the clam shell bucket is below a specified threshold
  which is initially estimated from the cutting edges of the supplied shovels.

  */
  class AGXTERRAIN_EXPORT ClamShellBucket : public agxTerrain::CompositeShovel
  {
  public:

    /**
    Find first shovel with given name.
    \param simulation - simulation the shovel is part of
    \param name - name of the shovel
    \return shovel if found - otherwise nullptr
    */
    static ClamShellBucket* find( const agxSDK::Simulation* simulation, const agx::Name& name );

    /**
    Find shovel with given UUID.
    \param simulation - simulation the shovel is part of
    \param uuid - UUID of the shovel
    \return shovel if found - otherwise nullptr
    */
    static ClamShellBucket* find( const agxSDK::Simulation* simulation, const agx::Uuid& uuid );

    /**
    Finds all shovels in the given simulation.
    \param simulation - simulation with shovels.
    \return vector of shovels
    */
    static ClamShellBucketPtrVector findAll( const agxSDK::Simulation* simulation );

    /**
    Enum specifing the available shovels in a ClamShellBucket.
    */
    enum class SHOVEL : agx::UInt16
    {
      FIRST = 0,
      SECOND = 1
    };

  public:
    /**
    Constructor.
    \param shovel1 - one of the shovels in the ClamShellBucket
    \param shovel2 - one of the shovels in the ClamShellBucket
    \note  - the closedThreshold is intially estimated from the
             cutting edges of the supplied shovels.
    */
    ClamShellBucket( agxTerrain::Shovel* shovel1,
                     agxTerrain::Shovel* shovel2 );

		/**
		\return one of the two shovels in the ClamShellBucket (0 or 1).
		*/
		Shovel* getShovel( SHOVEL i ) const;

    /**
    \return true if the clamshell bucket is in a closed state. (See: getClosedThreshold())
    */
    bool isClosed() const;

    /**
    \return true if the clamshell bucket is in a valid state.
    */
    virtual bool isValid() const override;

    /**
    The threshold value for the distance between the bucket shovel's
    cutting edges	that determines when the shovel is in a closed state.
    \param closedThreshold - the threshold to be used to determine when the
                             ClamShellBucket to be closed.
    */
    void setClosedThreshold( agx::Real closedThreshold );

    /**
    \return the threshold value that determines when the shovel is in a closed state.
    */
    agx::Real getClosedThreshold() const;

    /**
    \return the computed distance between the cutting edges of the shovels in the ClamShellBuckets.
    */
    agx::Real computeCurrentDistance() const;

    /*
    \return a vector containing the aggregate contacts between the shovel aggregates if the bucket is closed.
    */
    agxCollide::GeometryContactPtrVector getAggregateContacts( agxSDK::Simulation* simulation );

    /**
    \return the soil aggregate mass inside the clamshell bucket.
    */
    agx::Real getSoilMassInBucket() const;

    DOXYGEN_START_INTERNAL_BLOCK()

		/**
		Callback when this instance is added to a simulation.
		*/
		virtual void addNotification()  override;

		/**
		Callback when this instance is removed from a simulation.
		*/
		virtual void removeNotification() override;

		/**
		Pre collide callback.
		*/
		virtual void preCollide() override;

		/**
		Pre solver callback.
		*/
		virtual void pre() override;

		/**
		Post solver callback.
		*/
		virtual void post() override;

		/**
		Callback to be executed at the end of the time step
		*/
		virtual void last() override;

		ClamShellDebugRenderSettings* getDebugRenderSettings();
		AGXSTREAM_DECLARE_SERIALIZABLE( agxTerrain::ClamShellBucket );

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    ClamShellBucket();
    virtual ~ClamShellBucket();
    // Update closed state from the current edge distance
    void updateClosedState();
    // Update shovel settings from clamshell open/closed state
    void updateShovelSettings();
    // Add contacts between shovel aggregates if the clamshell is closed
    void updateAggregateContacts();

  protected:
    agx::Real                       m_closedThreshold;
    bool                            m_isClosed;
    ClamShellDebugRenderSettingsRef m_debugRenderSettings;
  };

  DOXYGEN_START_INTERNAL_BLOCK()

  /**
  Class that handles the debug render settings for the ClamShellBucket.
  */
  class AGXTERRAIN_EXPORT ClamShellDebugRenderSettings : public agx::Referenced
  {
  public:
    ClamShellDebugRenderSettings() : m_flags(), m_renderSize( 0.1 ){}

    void setRenderAggregateContacts( bool enable ) { m_flags.update( RENDER_AGGREGATE_CONTACTS, enable ); }
    bool getRenderAggregateContacts() const { return m_flags.Is( RENDER_AGGREGATE_CONTACTS ); }

    void setRenderMidPointDistance( bool enable ) { m_flags.update( RENDER_MIDPOINT_DISTANCE, enable ); }
    bool getRenderMidPointDistance() const { return m_flags.Is( RENDER_MIDPOINT_DISTANCE ); }

    agx::Real getRenderSize() const { return m_renderSize; }
    void setRenderSize(agx::Real size) { m_renderSize = size; }

  protected:
    virtual ~ClamShellDebugRenderSettings(){}

    enum StateFlags : agx::UInt32
    {
      RENDER_AGGREGATE_CONTACTS = 1 << 0,
      RENDER_MIDPOINT_DISTANCE = 1 << 1,
    };
    using Flags = agx::BitState<StateFlags, agx::UInt32>;

    Flags         m_flags;
    agx::Real     m_renderSize;
  };

  DOXYGEN_END_INTERNAL_BLOCK()
}
