/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SensorOutput.h>
#include <agxSensor/SystemNode.h>

#include <agx/Referenced.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( EncoderOutput );

  /**
  Encoder sensor output data where EncoderOutput::Field is used to define the data available.

  It's possible to pack data into a custom type, e.g.,

    struct MyRotaryEncoderOutput
    {
      agx::Real angle;
      agx::Real rate;
    }

    EncoderOutputRef output = new EncoderOutput();
    output->build<EncoderOutput::POSITION_F64,
                  EncoderOutput::SPEED_F64>();

    ... // Run sensor

    auto view = output->view<MyRotaryEncoderOutput>();
    for ( const MyRotaryEncoderOutput& value : view )
      ...
  */
  class AGXSENSOR_EXPORT EncoderOutput : public SystemNode, public ISensorOutput
  {
    public:
      enum Field : int32_t
      {
        /** 1x 64-bit floating point encoder output position. */
        POSITION_F64,
        /** 1x 64-bit floating point encoder output position change speed. */
        SPEED_F64,
        /** (Number of fields indicator) */
        NUM_FIELDS
      };

      using FieldVector = std::vector<Field>;

    public:
      /**
      \param field - data field
      \return data size of the field in bytes
      */
      static size_t sizeOf( Field field );

      /**
      \param field - data field
      \return name of the given field
      */
      static const char* getFieldName( Field field );

      /**
      Default constructor.
      */
      EncoderOutput() = default;

      /**
      Add one or more fields, build a matching data structure.
      */
      template<Field... field>
      void build();

      /**
      Add new field.
      \param field - field to add
      \return reference to this encoder output
      */
      EncoderOutput& add( Field field );

      /**
      "Element size" is the summed data size of the currently added fields.
      \return the current, summed, data size of the added fields
      */
      virtual size_t getElementSize() const override;

      /**
      \return IMU output data as a pure binary buffer
      */
      virtual const BinaryOutputBuffer& getData() override;

      /**
      \param markAsRead - true to reset unread state, resulting in hasUnreadData returning
                          false until new data is available. False to not mark the current
                          data as read.
      \return true when the data has been updated but not read/used/fetched by
              the user, false if the data has been read
      */
      virtual bool hasUnreadData( bool markAsRead = true ) override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~EncoderOutput() = default;

      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::EncoderOutput );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      FieldVector m_fields{};
      size_t m_elementSize{ 0u };
      bool m_hasUnreadData{ false };
      BinaryOutputBuffer m_output{};
  };

  template<EncoderOutput::Field... field>
  void EncoderOutput::build()
  {
    m_fields.clear();
    m_elementSize = 0u;
    ( this->add( field ), ... );
  }

  DOXYGEN_START_INTERNAL_BLOCK()  // swig

#pragma pack(push, 1)
  struct PositionSpeedValue
  {
      agx::Real position, speed;

      inline PositionSpeedValue() {}
      inline PositionSpeedValue(const agx::Vec2& v)
        : position(v.x())
        , speed(v.y())
      {
      }
      inline PositionSpeedValue(const PositionSpeedValue&) = default;
      inline PositionSpeedValue& operator=(const PositionSpeedValue&) = default;

      inline agx::Vec2 asVec2() const { return agx::Vec2(position, speed); }
  };
#pragma pack(pop)

  class AGXSENSOR_EXPORT EncoderOutputPosition : public EncoderOutput
  {
    public:
      EncoderOutputPosition();
  };

  class AGXSENSOR_EXPORT EncoderOutputSpeed : public EncoderOutput
  {
    public:
      EncoderOutputSpeed();
  };

  class AGXSENSOR_EXPORT EncoderOutputPositionSpeed : public EncoderOutput
  {
    public:
      EncoderOutputPositionSpeed();
  };

  DOXYGEN_END_INTERNAL_BLOCK()
}
