/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or
having been advised so by Algoryx Simulation AB for a time limited evaluation,
or having purchased a valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxModel/export.h>
#include <agxDriveTrain/Shaft.h>


namespace agxDriveTrain
{

  /**
  Parameters controlling the mean effective friction torque. This corresponds to
  page 722, section 13.5.1 in "Internal combustion engine fundamentals" by John B. Heywood, 1988.
  */
  struct AGXMODEL_EXPORT FrictionTorqueParameters
  {
    agx::Real cfr0;
    agx::Real cfr1;
    agx::Real cfr2;
    FrictionTorqueParameters();

    bool operator==(const FrictionTorqueParameters& other) const
    {
      return
        (cfr0 == other.cfr0 &&
          cfr1 == other.cfr1 &&
          cfr2 == other.cfr2);
    }
  };

  /**
  Parameters that specifies the performance of a combustion engine.
  */
  struct AGXMODEL_EXPORT CombustionEngineParameters
  {
    enum RevolutionsPerCycle
    {
      TWO_STROKE = 1,
      FOUR_STROKE = 2
    };


    agx::Real displacementVolume;               /**< The total displacement volume (m^3) of the engine, which is the sum of the volumes of the cylinders. */
    agx::Real maxTorque;                        /**< The maximum rated torque (Nm) which is the highest brake torque that an engine is allowed to deliver. */
    agx::Real maxTorqueRPM;                     /**< The rated torque speed (rpm) (i.e. the crankshaft rotational speed) in which the maximum rated torque is delivered. */
    agx::Real maxPowerRPM;                      /**< The rated power speed (rpm) (i.e. the crankshaft rotational speed) in which the maximum rated power is delivered. */
    agx::Real idleRPM;                          /**< The crankshaft speed (rpm) of the engine that is idling. */
    agx::Real maxRPM;                           /**< The maximum permissible crankshaft speed (RPM) of the engine. Exceeding this limit can cause engine damage or failure. In the simulation, to ensure safety, the engine will be automatically shut down upon reaching this RPM. */
    agx::Real crankShaftInertia;                /**< The moment of inertia (kg.m^2) of the crankshaft. */
    agx::Real throttlePinBoreRatio;             /**< The ratio between throttle pin diameter and throttle bore diameter. */
    agx::Real maxVolumetricEfficiency;          /**< Maximum volumetric efficiency of the engine. The volumetric efficiency refers to the ratio of air volume drawn into the cylinders to the volume the cylinder sweeps. */
    agx::Real airFuelRatio;                     /**< The stoichiometric air-fuel ratio. Ideal ratio of air to fuel needed for complete combustion, e.g. gasoline. */
    agx::Real heatValue;                        /**< The heat value of the fuel. */
    agx::Real idleThrottleAngle;                /**< The smallest angle the throttle plate can have when the engine is running at idle. */
    agx::Real maxThrottleAngle;                 /**< The maximum angle that the throttle plate is allowed to have. */
    RevolutionsPerCycle nrRevolutionsPerCycle;  /**< The number of crankshaft revolutions during one engine operating cycle. */
    agx::String reference;                      /**< Reference to documentation of this engine. */
    agx::String description;                    /**< Description of this engine. */
    agx::String name;                           /**< Name of this engine. */

    /// return true if all parameters are equal to p other.
    bool operator==(const CombustionEngineParameters& other) const;

    /**
    Default constructor. Will initialize the parameters for a Volvo-T5 diesel engine used in Volvo V60.
    */
    CombustionEngineParameters();


    /**
    Parameterizable constructor.

    \param displacementVolume_ - The total displacement volume (m^3) of the engine, which is the sum of the volumes of the cylinders.
    \param maxTorque_ - The maximum rated torque (Nm) which is the highest brake torque that an engine is allowed to deliver over short/continous periods of operations.
    \param maxTorqueRPM_  - The rated torque speed (rpm) (i.e. the crankshaft rotational speed) in which the maximum rated torque is delivered.
    \param maxPowerRPM_ - The rated power speed (rpm) (i.e. the crankshaft rotational speed) in which the maximum rated power is delivered.
    \param idleRPM_ - The crankshaft speed (rpm) of the engine that is idling.
    \param maxRPM_ - The maximum allowable crankshaft speed (rpm) of the engine.
    \param crankShaftInertia_ - The moment of inertia (kg.m^2) of the crankshaft.
    \param throttlePinBoreRatio_ - The ratio between throttle pin diameter and throttle bore diameter.
    \param maxVolumetricEfficiency_ - Maximum volumetric efficiency of the engine. The volumetric efficiency refers to the ratio of air volume drawn into the cylinders to the volume the cylinder sweeps.
    \param airFuelRatio_ - The stoichiometric air-fuel ratio. Ideal ratio of air to fuel needed for complete combustion, e.g. gasoline.
    \param heatValue_ - The heat value of the fuel.
    \param idleThrottleAngle_ - The smallest angle the throttle plate can have when the engine is running at idle.
    \param maxThrottleAngle_ - The maximum angle that the throttle plate is allowed to have.
    \param nrRevolutionsPerCycle_ - The number of crankshaft revolutions during one engine operating cycle.
    */
    CombustionEngineParameters(agx::Real displacementVolume_,
      agx::Real maxTorque_,
      agx::Real maxTorqueRPM_,
      agx::Real maxPowerRPM_,
      agx::Real idleRPM_,
      agx::Real maxRPM_,
      agx::Real crankShaftInertia_,
      agx::Real throttlePinBoreRatio_ = 0.1,
      agx::Real maxVolumetricEfficiency_ = 0.92,
      agx::Real airFuelRatio_ = 14.6,
      agx::Real heatValue_ = 42500000,
      agx::Real idleThrottleAngle_ = agx::Real(10 * agx::DEG_TO_RAD),
      agx::Real maxThrottleAngle_ = agx::Real(agx::PI_2),
      RevolutionsPerCycle nrRevolutionsPerCycle_ = RevolutionsPerCycle::FOUR_STROKE);

    FrictionTorqueParameters frictionTorqueParameters;
  };

}
