/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/IGenericConstraint.h>
#include <agx/GenericConstraintData.h>

namespace agx
{
  /**
  Generic many-body constraint with callbacks for writing Jacobian
  and constraint data. This generic many-body constraint works as
  an elementary constraint but may have 'child' constraints instead
  of additional elementary constraints. It's also possible to implement
  elementary constraints for many-body constraints.

  This constraint receives; addNotification, removeNotification, onPreSolve,
                            onWriteData and onPostSolve.

  To access Jacobian data during onWriteData - getBodyData( body ) is used
  and getBodyData may return nullptr when the Jacobian should be ignored.
  Note that addJacobian( translational, rotational ) may be used instead
  of setJacobian if the body appears several times.

  Any structural changes, e.g., add remove of bodies or child constraints
  must be made before the onWriteData callback (onPreSolve or pre-collide/pre
  step callbacks).

  This constraint may be empty of bodies and data, serving as a parent of
  many other constraints.

  Python example:
    class MyWireSegment(agx.GenericManyBodyConstraint):
        def __init__(self, nodes):
            super().__init__(numRows=1)
            self.nodes = nodes

        def onPreSolve(self):
            self.removeAllBodies()
            num_bodies = 0
            for node in self.nodes:
                num_bodies += 1 if self.add(node.body) else 0
            return num_bodies > 0

      def onWriteData(self, constraint_data: agx.GenericConstraintData):
          for node in self.nodes:
              data = constraint_data.getBodyData(node.body)
              if data is None:
                  continue
              # write Jacobian data
          constraint_data.setViolation(0, calculated_violation)


    def MyWire(agx.GenericManyBodyConstraint):
        def __init__(self, **kwargs):
            super().__init__(numRows=0)

        def addNotification(self):
            # Added to simulation - initialize.
            pass

        def onPreSolve(self):
            self.removeAllChildren()

            if len( self.nodes ) < 2:
              return False

            node = self.nodes.front
            while node:
                segment, node = self._create_segment(node)
                if segment:
                  self.addChild(segment)
  */
  class AGXPHYSICS_EXPORT GenericManyBodyConstraint : public IGenericConstraint
  {
    public:
      /**
      Construct given number of rows.
      \param numRows - number of rows
      */
      GenericManyBodyConstraint( agx::UInt numRows );

    public:
      /**
      Callback when the constraint has been added to a simulation.
      */
      virtual void onAddNotification() override;

      /**
      Callback when the constraint has been removed from a simulation.
      */
      virtual void onRemoveNotification() override;

      /**
      Callback before this constraint reaches the solver (actually preSystemCallback). If the
      return value is true and this constraint is valid, this constraint will continue into
      the solver.
      \return true if this constraint is active, false if inactive
      */
      virtual agx::Bool onPreSolve() override;

      /**
      Callback when to write Jacobian and constraint data if this constraint is active.
      \param constraintData - constraint data for this elementary constraint
      */
      virtual void onWriteData( agx::GenericConstraintData* constraintData );

      /**
      Callback from after the solve and the solutions been pushed to the elementary constraints.
      */
      virtual void onPostSolve() override;

      /**
      Callback to write data. All values must be greater than or equal to zero. Data is
      only stored for the lower triangle since the matrix must be symmetric and positive
      definite.
      */
      virtual void updateComplianceMatrix( GenericConstraintComplianceMatrix* matrix ) override;

    protected:
      virtual ~GenericManyBodyConstraint() = default;
  };
}
