
#pragma once

#include <algorithm>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <unordered_map>
#include <unordered_set>
#include <openplx/export.h>
#include <openplx/Any.h>
#include <openplx/ModelDeclaration.h>

namespace openplx {
    class RuntimeContext;
}

#ifdef _MSC_VER
#  pragma warning(push)
#  pragma warning(disable: 4251)
#endif

namespace openplx::Core
{
    using ObjectPtr = std::shared_ptr<Object>;
    class ExpressionEvaluator;
    /**
     * @brief root class of all models, both bundle models and dynamic models
     *
     */
    class OPENPLX_CORE_EXPORT Object {
        public:
            Object();
            explicit Object(std::string name);
            virtual ~Object() = default;
            Object(const Object &) = delete;

            /**
             * Virtual function that is called (via triggerOnInit) on a Object
             * after evaluation is completed.
             */
            virtual void on_init(const openplx::RuntimeContext& context);
            virtual void setDynamic(const std::string& key, Any&& value);
            virtual void extractObjectFieldsTo(std::vector<std::shared_ptr<Object>>& output) const;
            void extractNestedObjectFieldsTo(std::unordered_set<std::shared_ptr<Object>>& output) const;
            virtual void extractEntriesTo(std::vector<std::pair<std::string, Any>>& output) const;

            virtual Any getDynamic(const std::string& key) const;
            virtual Any callDynamic(const std::string& key, const std::vector<Any>& args);

            template <class T>
            T get(const std::string& key) const;

            template <class T>
            bool is() const {
                return dynamic_cast<const T*>(this) != nullptr;
            }
            template <class T>
            const T& as() const {
                return *dynamic_cast<const T*>(this);
            }


            const std::string& getName() const;
            const std::string& getUuid() const;

            /**
             * Returns all the Objects attribute values of the specified type T
             */
            template <class T>
            std::vector<std::shared_ptr<T>> getValues() const {
                std::vector<std::shared_ptr<T>> output;
                std::vector<std::shared_ptr<Object>> all_fields;
                this->extractObjectFieldsTo(all_fields);
                for (auto& field : all_fields) {
                    auto object = std::dynamic_pointer_cast<T>(field);
                    if (object == nullptr) {
                        continue;
                    }
                    // See openplx issue 403. By introducing references we could
                    // force models not to have more than one attribute with the same pointer value
                    // unless all but one are references.
                    if (std::find<typename std::vector<std::shared_ptr<T>>::const_iterator>(output.cbegin(), output.cend(), object) == output.cend()) {
                        output.push_back(object);
                    }
                }
                return output;
            }

            /**
             * Returns all the Objects attribute values of the specified type T
             */
            template <class T>
            std::vector<std::shared_ptr<T>> getNonReferenceValues() const {
                std::vector<std::shared_ptr<T>> output;
                auto entries = getNonReferenceEntries<T>();
                for (auto& entry : entries) {
                    output.push_back(entry.second);
                }
                return output;
            }

            /**
             * Return all attribute name,value pairs of the specified type T
             */
            template <class T>
            std::vector<std::pair<std::string, std::shared_ptr<T>>> getEntries() const {
                std::vector<std::pair<std::string, std::shared_ptr<T>>> output;
                std::vector<std::pair<std::string, Any>> all_entries;
                this->extractEntriesTo(all_entries);
                for (auto& entry : all_entries) {
                    if (!entry.second.isObject())
                        continue;
                    auto object = std::dynamic_pointer_cast<T>(entry.second.asObject());
                    if (object == nullptr) {
                        continue;
                    }
                    output.push_back({entry.first, object});
                }
                return output;
            }

            /**

             * Return all attribute {name, value} pairs of the specified type T
             */
            template <class T>
            std::vector<std::pair<std::string, std::shared_ptr<T>>> getNonReferenceEntries() const {
                std::vector<std::pair<std::string, std::shared_ptr<T>>> output;
                std::vector<std::pair<std::string, Any>> all_entries;
                this->extractEntriesTo(all_entries);
                for (auto& entry : all_entries) {
                    if (!entry.second.isObject() || entry.second.isReference())
                        continue;
                    auto object = std::dynamic_pointer_cast<T>(entry.second.asObject());
                    if (object == nullptr || (object->getOwner() != nullptr && object->getOwner().get() != this)) {
                        continue;
                    }
                    output.push_back({entry.first, object});
                }
                return output;
            }

            /**
             * Return a flat representation of the tree of all Object attributes of type T and their respective Object attributes of Type T, and so forth
             */
            template <class T>
            std::vector<std::shared_ptr<T>> getNestedObjects() const {
                std::vector<std::shared_ptr<T>> output;
                std::unordered_set<std::shared_ptr<Object>> all_objects;
                this->extractNestedObjectFieldsTo(all_objects);
                for (auto& obj : all_objects) {
                    auto target = std::dynamic_pointer_cast<T>(obj);
                    if (target == nullptr) {
                        continue;
                    }
                    output.push_back(target);
                }
                return output;
            }

            /**
             * Virtual function that first calls triggerOnInit on all children
             * before calling on_init on itself.
             */
            virtual void triggerOnInit(const openplx::RuntimeContext& context);


            ModelDeclPtr getType() const;

            const std::vector<std::string>& getTypeList() const;


            ObjectPtr getOwner() const;

            /**
             * Compares two object by type and primitive parameters
             * @return true if the two objects type lists are
             *         identical and all primitive attributes are identical
             *         and neither of them contains a non-primitive attribute,
             *         false otherwise.
             **/
            static bool compareByTypeAndPrimitiveAttributes(Object& obj1, Object& obj2);

            /**
             * Returns true if the value of a field was set from the right hand side
             * of the declaring "var declaration". It is default in the sense that it is
             * the value you will get if you don't modify the model by overriding the
             * value with a "var assignment".
             */
            bool isDefault(const std::string& key) const;
            void setAsDefault(const std::string& key);

            /* Annotations */
            void appendToAnnotations(AnnotationPtr annotation);
            const std::vector<AnnotationPtr>& getAnnotations() const;
            std::vector<AnnotationPtr> findAnnotations(const std::string& name) const;

            /* Traits */
            void appendToTraits(std::string trait);
            const std::unordered_set<std::string>& getTraits() const;
            bool hasTrait(std::string trait) const;

            /* Utility methods */

            /**
             * Queries an object for a member object, can take a '.' separated
             * string to access nested objects.
             * @return nullptr if no object was found, the object otherwise
             */
            ObjectPtr getObject(const std::string& key);

            /**
             * Queries an object for a nested key
             * @return An Any representing the value or undefined if the key did not match anything
             */
            Any getAny(const std::string& key);

            /**
             * Queries an object for numeric (real or int), string, or bool respectively.
             * Throws a runtime_error if the key is not found or if it is found but does
             * not match the requested type. Can take a '.' seperated string to access
             * nested symbols.
             */
            double getNumber(const std::string& key);
            std::string getString(const std::string& key);
            bool getBool(const std::string& key);

            /**
             * Serializes the object to unformatted json
             * @param follow_refs If true references to other openplx::Core::Object:s will also be
             *        serialized, if false references will serialized only as a dummy
             *        openplx::Core::Object with uuid.
             */
            std::string toJson(bool follow_refs = false);

        protected:
            std::vector<std::string> m_type_list;
            bool m_initialized;

        private:
            void setName(const std::string& name);
            void writeMembersAsJson(std::ostringstream& out, std::unordered_set<Object*>& loop_detector, bool follow_refs);
            void writeValueAsJson(std::ostringstream& out, Any& value, std::unordered_set<Object*>& loop_detector, bool follow_refs);

            std::unordered_map<std::string, Any> m_dynamic_values;
            std::vector<AnnotationPtr> m_annotations;
            std::unordered_set<std::string> m_default_keys;
            std::unordered_set<std::string> m_traits;
            ModelDeclPtr m_model_decl;
            std::string m_name;
            std::string m_uuid;
            std::weak_ptr<Object> m_owner; // The instance that held this instance first

        friend class ExpressionEvaluator;
        friend class Evaluator;
    };

    // Used in generated bundles
    using DynamicMethodPointer = Any (*)(openplx::Core::Object*, std::vector<Any>);
}

#ifdef _MSC_VER
#  pragma warning(pop)
#endif
