
#pragma once

#include <memory>
#include <optional>
#include <unordered_map>
#include <unordered_set>
#include <openplx/Object.h>

namespace openplx::Core {

    using FactoryMethod = std::shared_ptr<Object> (*)();
    using StaticMethod = Any (*)(std::vector<Any>);
    using BinaryOperatorMethod = Any(*)(Any lhs, Any rhs);
    using UnaryOperatorMethod = Any(*)(Any operand);

    class Evaluator;

    class EvaluatorContext {
        public:
            inline void registerFactory(const char * className, FactoryMethod createMethod) {
                m_registeredFactories[className] = createMethod;
            }
            inline void registerStaticMethod(const char * key, StaticMethod static_method) {
                m_staticMethods[key] = static_method;
            }
            inline void registerBinaryOperatorMethod(const char * key, BinaryOperatorMethod bin_op_method) {
                m_binaryOperatorMethods[key] = bin_op_method;
            }
            inline void registerUnaryOperatorMethod(const char * key, UnaryOperatorMethod unary_op_method) {
                m_unaryOperatorMethods[key] = unary_op_method;
            }
            inline void registerObject(openplx::Core::ObjectPtr object) {
                m_objects.insert(std::move(object));
            }
            inline std::optional<FactoryMethod> lookup(std::string className) {
                auto search = m_registeredFactories.find(className);
                if (search == m_registeredFactories.end()) {
                    return std::nullopt;
                }
                return search->second;
            }
            inline Any callStaticMethod(std::string key, std::vector<Any> args) {
                auto iter = m_staticMethods.find(key);
                if (iter == m_staticMethods.end()) {
                    return Any();
                }
                return iter->second(args);
            }
            inline Any callBinaryOperator(std::string key, Any lhs, Any rhs) {
                auto iter = m_binaryOperatorMethods.find(key);
                if (iter == m_binaryOperatorMethods.end()) {
                    return Any();
                }
                return iter->second(lhs, rhs);
            }
            inline Any callUnaryOperator(std::string key, Any operand) {
                auto iter = m_unaryOperatorMethods.find(key);
                if (iter == m_unaryOperatorMethods.end()) {
                    return Any();
                }
                return iter->second(operand);
            }
            inline const std::unordered_set<openplx::Core::ObjectPtr>& getRegisteredObjects() const {
                return m_objects;
            }

            inline void clear() {
                m_objects.clear();
            }

        private:
            std::unordered_map<std::string, FactoryMethod> m_registeredFactories;
            std::unordered_map<std::string, StaticMethod> m_staticMethods;
            std::unordered_map<std::string, BinaryOperatorMethod> m_binaryOperatorMethods;
            std::unordered_map<std::string, UnaryOperatorMethod> m_unaryOperatorMethods;

            // Keeps track of all created objects
            std::unordered_set<openplx::Core::ObjectPtr> m_objects;

            friend class openplx::Core::Evaluator;
    };
}
