/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxTerrain/export.h>
#include <agxCollide/BoundingAABB.h>
#include <agxRender/Color.h>
#include <agx/AffineMatrix4x4.h>
#include <agx/Frame.h>
#include <agx/BitState.h>
#include <agx/Vector.h>
#include <agx/Referenced.h>


namespace agxTerrain
{
  class Terrain;

  AGX_DECLARE_POINTER_TYPES( IForbiddenBound );
  AGX_DECLARE_VECTOR_TYPES( IForbiddenBound );
  /**
  Base abstract class for specifying different types of bounds bound
  where terrain operations like merging and avalanching is forbbiden.
  */
  class AGXTERRAIN_EXPORT IForbiddenBound : public agx::Referenced, public agxStream::Serializable
  {
  public:
    /**
    Checks if the specified terrain index is contained
    in the ForbiddenBound for a specified terrain.
    \param terrainIndex - The specified index to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the index to a position that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsIndex( const agx::Vec3i& terrainIndex, agxTerrain::Terrain* terrain ) const = 0;

    /**
    Checks if the specified world position is contained
    in the ForbiddenBound for a specified terrain.
    \param position - The specified world position to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the position to a local terrain index that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsPosition( const agx::Vec3& position, agxTerrain::Terrain* terrain ) const = 0;


    virtual void debugRender( agxTerrain::Terrain* terrain,
                              agx::Real size,
                              const agxRender::Color& color,
                              bool isStatic ) const = 0;

    AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( IForbiddenBound );

    /**
    Set enable of the forbidden bound. If false all contains checks will return false.
    */
    void setEnable( bool enable ) { m_enable = enable; }

    /**
    Get enable of the forbidden bound. If false all contains checks will return false.
    */
    bool getEnable() const { return m_enable; }

    /**
    Set enable of debug rendering of the forbidden bound. All debugRender() calls will produce
    nothing if this is disabled.
    */
    void setEnableDebugRendering( bool enable ) { m_enableDebugRendering = enable; }

    /**
    \return if debug rendering of the forbidden bound. All debugRender() calls will produce
    nothing if this is disabled.
    */
    bool getEnableDebugRendering() const { return m_enableDebugRendering; }

  protected:
    IForbiddenBound();
    virtual ~IForbiddenBound() = default;

    /**
    Restore.
    Abstract serializable, so child classes must explicitly call these methods.
    */
    void restore( agxStream::InputArchive& in ) override;

    /**
    Store.
    Abstract serializable, so child classes must explicitly call these methods.
    */
    void store( agxStream::OutputArchive& out ) const override;

  protected:
    bool m_enable;
    // We do not serialize this variable
    bool m_enableDebugRendering;
  };



  AGX_DECLARE_POINTER_TYPES( ForbiddenBound );
  /**
  Class for specifying a 3D bound where terrain operations like
  merging and avalanching is forbbiden.
  */
  class AGXTERRAIN_EXPORT ForbiddenBound : public IForbiddenBound
  {
  public:
    /**
    Constructor for a 3D position ForbiddenBound.
    \param bound - The specified world position Axis-Alinged bound that will be used.
    \param localTransform - The local transform of the bound. This can be used to arbitrary translate
                            and rotate the axis alinged bound structure.
    \param parentTransform - A parent transform than can be supplied to make the bound follow a dynamically updated
                             transform, such as one from a agxCollide::Geometry or agx::RigidBody.
    */
    ForbiddenBound(
      const agxCollide::BoundingAABB& bound,
      agx::AffineMatrix4x4 localTransform = agx::AffineMatrix4x4(),
      agx::Frame* parentTransform = nullptr );

    /**
    Checks if the specified terrain index is contained
    in the ForbiddenBound for a specified terrain.
    \param terrainIndex - The specified index to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the index to a position that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsIndex( const agx::Vec3i& terrainIndex, agxTerrain::Terrain* terrain ) const override;

    /**
    Checks if the specified world position is contained
    in the ForbiddenBound for a specified terrain.
    \param position - The specified world position to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the position to a local terrain index that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsPosition( const agx::Vec3& position, agxTerrain::Terrain* terrain ) const override;

    /// \return the internal bound.
    agxCollide::BoundingAABB getBound() const { return m_bound; }

    /// \return the parent frame.
    agx::Frame* getParentFrame() const { return m_parentTransform; }

    /// \return the local transform.
    const agx::AffineMatrix4x4& getLocalTransform() const { return m_localTransform; }

    DOXYGEN_START_INTERNAL_BLOCK()

      virtual void debugRender( agxTerrain::Terrain* terrain,
      agx::Real size,
      const agxRender::Color& color,
      bool isStatic ) const override;

    AGXSTREAM_DECLARE_SERIALIZABLE( ForbiddenBound );

    DOXYGEN_END_INTERNAL_BLOCK()
  protected:
    ForbiddenBound();
    virtual ~ForbiddenBound() = default;

  protected:
    agxCollide::BoundingAABB  m_bound;
    agx::AffineMatrix4x4      m_localTransform;
    agx::AffineMatrix4x4      m_invLocalTransform;
    agx::FrameRef             m_parentTransform;
  };



  AGX_DECLARE_POINTER_TYPES( ForbiddenIndexBound );
  /**
  Class for specifying a 3D terrain index bound where terrain operations like
  merging and avalanching is forbbiden.
  */
  class AGXTERRAIN_EXPORT ForbiddenIndexBound : public IForbiddenBound
  {
  public:
    /**
    Constructor for a 3D index ForbiddenBound.
    \param bound - The specified local terrain index Axis-Alinged bound that will be used.
    \note containsPosition uses the supplied terrain to convert the world position
          to a local terrain index.
    */
    ForbiddenIndexBound( const agx::Bound3i& bound );

    /**
    Checks if the specified terrain index is contained
    in the ForbiddenBound for a specified terrain.
    \param terrainIndex - The specified index to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the index to a position that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsIndex( const agx::Vec3i& terrainIndex, agxTerrain::Terrain* terrain ) const override;

    /**
    Checks if the specified world position is contained
    in the ForbiddenBound for a specified terrain.
    \param position - The specified world position to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the position to a local terrain index that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsPosition( const agx::Vec3& position, agxTerrain::Terrain* terrain ) const override;

    /// \return the internal bound.
    agx::Bound3i getBound() const { return m_bound; }

    DOXYGEN_START_INTERNAL_BLOCK()

    virtual void debugRender( agxTerrain::Terrain* terrain,
                              agx::Real size,
                              const agxRender::Color& color,
                              bool isStatic ) const override;

    AGXSTREAM_DECLARE_SERIALIZABLE( ForbiddenIndexBound );

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    ForbiddenIndexBound();
    virtual ~ForbiddenIndexBound() = default;

  protected:
    agx::Bound3i m_bound;
  };



  AGX_DECLARE_POINTER_TYPES( ForbiddenIndex2DBound );
  /**
  Class for specifying a 2D terrain index bound where terrain operations like
  merging and avalanching is forbbiden.
  */
  class AGXTERRAIN_EXPORT ForbiddenIndex2DBound : public IForbiddenBound
  {
  public:
    /**
    Constructor for a 2D index ForbiddenBound.
    \param bound - The specified local terrain index 2D Axis-Alinged bound that will be used.
    \note containsPosition uses the supplied terrain to convert the world position
          to a local terrain index.
    */
    ForbiddenIndex2DBound( const agx::Bound2i& bound );

    /**
    Checks if the specified terrain index is contained
    in the ForbiddenBound for a specified terrain.
    \param terrainIndex - The specified 3D index to test against the 2D bound. Note
                          that only the x and y components of the specified index
                          will be used in the check.
    \param terrain - the terrain object that will be used to
                     transform the index to a position that can
                     be tested against the specified bound, if nessecary.
    \note only the x and y components of the supplied 3D index will be used in the bounds checking.
    */
    virtual bool containsIndex( const agx::Vec3i& terrainIndex, agxTerrain::Terrain* terrain ) const override;

    /**
    Checks if the specified world position is contained
    in the ForbiddenBound for a specified terrain.
    \param position - The specified world position to test against bound.
    \param terrain - the terrain object that will be used to
                     transform the position to a local terrain index that can
                     be tested against the specified bound, if nessecary.
    */
    virtual bool containsPosition( const agx::Vec3& position, agxTerrain::Terrain* terrain ) const override;


    /// \return the internal bound.
    agx::Bound2i getBound() const { return m_bound; }

    DOXYGEN_START_INTERNAL_BLOCK();

      virtual void debugRender( agxTerrain::Terrain* terrain,
        agx::Real size,
        const agxRender::Color& color,
        bool isStatic ) const override;

    AGXSTREAM_DECLARE_SERIALIZABLE( ForbiddenIndex2DBound );

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:
    ForbiddenIndex2DBound();
    virtual ~ForbiddenIndex2DBound() = default;

  protected:
    agx::Bound2i m_bound;
  };
}
