/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#include <agx/BitState.h>
#include <agxTerrain/SoilParticleAggregate.h>
#include <agxTerrain/SoilPenetrationResistance.h>
#include <agxTerrain/AggregateContactGenerator.h>
#include <agxTerrain/DeformController.h>
#include <agxTerrain/ShovelSettings.h>
#include <agxStream/Serializable.h>
#include <agx/Plane.h>

#pragma once

namespace agxCollide
{
    class Geometry;
    class Shape;
}

namespace agxTerrain
{
    //--- Forward declarations ---//
    class Terrain;
    class TerrainMaterial;

    using TerrainPtrVec = agx::VectorPOD<Terrain*>;

    AGX_DECLARE_POINTER_TYPES(ExternalDeformer);
    AGX_DECLARE_VECTOR_TYPES(ExternalDeformer);

    /**
    ExternaDeformer is a TerrainToolInstance class that models deformer surfaces
    that are not attached to a shovel object. Deformers are used to have surfaces
    push and deform terrain (but not dig in the terrain).

    A shovel object has three deformers on the bottom and two sides. However, the
    ExternalDeformer object can be attached to any surface that should be deforming
    the terrain when the surface is being pushed against the terrain. For example
    the sides of a track model that is turning, or any heavy objects that are being
    pushed in the soil.
    */
    class AGXTERRAIN_EXPORT ExternalDeformer : public agxSDK::TerrainToolInstance
    {
    public:
        class ExcavationSettings;
        class Settings;

        /**
        Basic constructor.
        \param parentBody - The parent body of the external deformer.
        \param cuttingEdge - The lowest edge on the deformer, specified in the LOCAL parentBody frame
                             that serves as the cutting edge of the active zone.
        \param topEdge - The top edge of the active zone, specified in the LOCAL parentBody frame
                         that will be created in front of the deformer.
        \param cuttingDirection - The cutting direction of the deformer.
        */
        ExternalDeformer( agx::RigidBody* parentBody,
                          const agx::Line& cuttingEdge,
                          const agx::Line& topEdge,
                          const agx::Vec3& cuttingDirection );

        /**
        Empty constructor, for creating a external deformer body to which can be added multiple deformers later.
        */
        ExternalDeformer(agx::RigidBody* parentBody);

        /**
        Add new deformer.
        */
        void addDeformer( const agx::Line& cuttingEdge,
                          const agx::Line& topEdge,
                          const agx::Vec3& cuttingDirection );

        /**
        \return the base rigid body from the ExternalDeformer object.
        */
        agx::RigidBody* getRigidBody() const;

        /**
        \return the deformer geometries colliding with voxels in the terrain
        */
        const agxCollide::GeometryRefVector& getVoxelCollisionGeometries() const;

        /**
        \return aggregate contact generator specific for this deformer instance
        */
        DeformController* getDeformController() const;

        /*
        \return the active terrain that the external deformer is acting on. If the external deformer is currently
        not interacting with the terrain, return nullptr.
        */
        Terrain* getActiveTerrain() const;

        /**
        Check if any of the collision geometries are inside terrain bounds
        */
        bool collisionGeometriesAreInsideTerrain(Terrain* terrain) const;

        /*
        \return if the deformer is currently active. The deformer is active while interacting with a terrain.
        */
        bool isActive() const;

        /*
        \return if the deformer is valid after construction. If it is not valid, it will not be possible to add the deformer to
        a simulation.
        */
        virtual bool isValid() const override;

        /*
        Set whether the external deformer should be enabled or not.
        \param enable - true if the external deformer is enabled, false otherwise.
        */
        void setEnable( bool enable );

        /*
        return whether the external deformer should is enabled or not.
        */
        bool getEnable() const;

        /**
        Set whenever debug rendering of deformer forbidden bounds should be
        enabled or not.
        \param enable - true if debug rendering should be enabled, false otherwise.
        */
        void setEnableDebugRendering( bool enable );

        /**
        \return whenever debug rendering should be used with external deformers or not.
        */
        bool getEnableDebugRendering() const;

        DOXYGEN_START_INTERNAL_BLOCK()

        virtual void onAssemblyAdd(agxSDK::Assembly* assembly) override;

        AGXSTREAM_DECLARE_SERIALIZABLE(agxTerrain::ExternalDeformer);

        DOXYGEN_END_INTERNAL_BLOCK()

    protected:
        /**
        Default constructor used in serialization.
        */
        ExternalDeformer();

        /**
        Reference counted object - protected destructor.
        */
        virtual ~ExternalDeformer();

        /*
        Creates a frame which uses the parents body's frame as parent. Other internals
        attaches to this frame to make sure they follow along the parent rigid body.
        Requires that the parent body isn't nullptr.
        */
        agx::Frame* createParentFrame();

        /**
        Add notification when a deformer is added to a simulation.
        */
        virtual void addNotification()  override;

        /**
        Remove notification when this deformer is removed from a simulation.
        */
        virtual void removeNotification() override;

        /**
        Set the common material used by the SoilParticleAggregates inside the deformers.
        Propagates the material into the internal classes.
        */
        void setAggregateMaterial(agx::Material* material);

        /*
        Checks if all internal contact materials are correct.
        If there is an active terrain (ie, we are interacting with a terrain)
        the contact material will be customized for that terrain.
        */
        void checkContactMaterials();

        /**
        Initialize contact materials when added to simulation
        */
        void initalizeContactMaterialsForExistingTerrains(agxSDK::Simulation* simulation);

        /**
        Executes pre-collide events for agxTerrain::ExternalDeformer in the simulation. Deformer and Terrain step events are
        synchronized via the TerrainManager. In the preCollide step, we:
        - Check if the deformer is active by colliding with a terrain.
        - If we need to, we create soil wedges in the deformers. (onPreCollide calls)
        */
        virtual void preCollide() override;

        /**
        Executes pre-step events for agxTerrain::ExternalDeformer in the simulation.
        Deformer and Terrain step events are synchronized via the TerrainManager.
        In the pre step we:
        - Filter contact points for the deformers
        - Calculate dead load fraction
        - Check and if need be, set the internal contact materials (this is a new ContactMaterial and transfer each pre step)
        - Run the onPre calls in all internal classes (aggregates, deformers, pen resistance)
        */
        virtual void pre() override;

        /**
        Executes post-step events for agxTerrain::ExternalDeformer in the simulation.
        Deformer and Terrain step events are synchronized via the TerrainManager.
        */
        virtual void post() override;

        /**
        Callback to be executed at the end of the time step
        Deformer and Terrain step events are synchronized via the TerrainManager.
        */
        virtual void last() override;

        agxSDK::TerrainManager* getTerrainManager();

        TerrainPtrVec getTerrainsInSimulation();

        /**
        Clear the active zone wedges of the deformer active zones.
        */
        void clearActiveZoneWedges() const;

    private:
        enum StateFlags : agx::UInt32
        {
          ENABLED = 1 << 0
        };
        using Flags = agx::BitState<StateFlags, agx::UInt32>;


        void addDeformers(const std::array<DeformerCollectionRef, 3>& deformers);

        bool shouldDeformMass(Terrain* terrain);

    protected:

        // Configuration:
        agx::RigidBodyRef                           m_parentBody;
        agx::FrameRef                               m_parentFrame; /* Parent frame for parent body geometries synchronized with model center of parent body. */
        Flags                                       m_flags;
        DeformControllerRef                         m_deformController;

        // Geometries used to find deformer-voxel collisions
        agx::MaterialRef                            m_aggregateMaterial;
        agxCollide::GeometryRefVector               m_voxelCollisionGeometries;
        Terrain*                                    m_activeTerrain;
        bool                                        m_isActive;
        agxCollide::BoundingAABB                    m_collisionBound;


        agx::Vector<DeformerCollectionRef>          m_deformers;

        bool                                        m_debugRenderForbiddenBound;
    };
}