/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceMaterial.h>
#include <agxSensor/EnvironmentInternalData.h>

#include <agxTerrain/Terrain.h>
#include <agxTerrain/TerrainPager.h>

#include <agxWire/Wire.h>

#include <agxSDK/LinkedStructure.h>

namespace agxSensor
{
  /**
  Surface material of shape instances in the raytrace environment.
  */
  class AGXSENSOR_EXPORT RtSurfaceMaterial : public RtMaterial
  {
    public:
      /**
      Assign/associate a surface material to the given shape. If/when the shape is
      added to a raytrace environment, \p surfaceMaterial will be used.
      \param shape - shape to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxCollide::Shape* shape, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given geometry. All shapes of the given
      geometry will inherit \p surfaceMaterial unless they haven't been explicitly
      assigned/associated a material.
      \param geometry - geometry to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxCollide::Geometry* geometry, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given rigid body. All geometries
      (and their shapes) of the given rigid body will inherit \p surfaceMaterial
      unless they haven't been explicitly assigned/associated a material.
      \param rb - rigid body to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agx::RigidBody* rb, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given linked structure (agxCable::Cable,
      agxModel::Beam, agxVehicle::Track, ...).
      \param linkedStructure - linked structure to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxSDK::LinkedStructure* linkedStructure, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given wire.
      \param wire - wire to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxWire::Wire* wire, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given terrain. The particles/granulars
      created from this terrain will also inherit the given material.
      \param terrain - terrain to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxTerrain::Terrain* terrain, RtSurfaceMaterial surfaceMaterial );

      /**
      Assign/associate a surface material to the given terrain (pager). The particles/granulars
      created from terrain of this pager will also inherit the given material.
      \param terrainPager - terrain pager to use the given \p surfaceMaterial
      \param surfaceMaterial - surface material instance
      */
      static void set( agxTerrain::TerrainPager* terrainPager, RtSurfaceMaterial surfaceMaterial );

      /**
      \return the surface material associated with the given shape
      */
      template<typename T, typename InstanceT>
      static T getAs( const InstanceT* instance );

    public:
      /**
      Assign/associate a surface material to the given shape. If/when the shape is
      added to a raytrace environment, \p surfaceMaterial will be used.
      \param shape - shape to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxCollide::Shape* shape ) const;

      /**
      Assign/associate a surface material to the given geometry. All shapes of the given
      geometry will inherit \p surfaceMaterial unless they haven't been explicitly
      assigned/associated a material.
      \param geometry - geometry to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxCollide::Geometry* geometry ) const;

      /**
      Assign/associate a surface material to the given rigid body. All geometries
      (and their shapes) of the given rigid body will inherit \p surfaceMaterial
      unless they haven't been explicitly assigned/associated a material.
      \param rb - rigid body to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agx::RigidBody* rb ) const;

      /**
      Assign/associate a surface material to the given linked structure (agxCable::Cable,
      agxModel::Beam, agxVehicle::Track, ...).
      \param linkedStructure - linked structure to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxSDK::LinkedStructure* linkedStructure ) const;

      /**
      Assign/associate a surface material to the given wire.
      \param wire - wire to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxWire::Wire* wire ) const;

      /**
      Assign/associate a surface material to the given terrain. The particles/granulars
      created from this terrain will also inherit the given material.
      \param terrain - terrain to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxTerrain::Terrain* terrain ) const;

      /**
      Assign/associate a surface material to the given terrain (pager). The particles/granulars
      created from terrain of this pager will also inherit the given material.
      \param terrainPager - terrain pager to use the given \p surfaceMaterial
      */
      RtSurfaceMaterial assignTo( agxTerrain::TerrainPager* terrainPager ) const;

      template<typename T>
      RtSurfaceMaterial assignTo( T* instance ) const
      {
        set( instance, *this );
        return *this;
      }

    protected:
      using RtMaterial::RtMaterial;
  };

  template<typename T, typename InstanceT>
  T RtSurfaceMaterial::getAs( const InstanceT* instance )
  {
    return T{ getRtMaterial( instance ) };
  }
}
