/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceSystemNode.h>
#include <agxSensor/SensorOutput.h>

#include <agx/AtomicState.h>
#include <agx/Logger.h>
#include <agx/Vec4.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( RtOutput );

  /**
  Raytrace output data where RtOutput::Field defines the data
  available.

  It's possible to pack data into a custom type, e.g.,

    struct RayPointAndDistance : agx::Vec4f
    {
      agx::Vec3f getPosition() const
      {
        // First three elements is the point.
        return { (*this)[ 0 ], (*this)[ 1 ], (*this)[ 2 ] };
      }

      float getDistance() const
      {
        // Fourth element is the distance.
        return (*this)[ 3 ];
      }
    };

    RtOutputRef pointAndDistanceOutput = new RtOutput();
    pointAndDistanceOutput->build<RtOutput::XYZ_VEC3_F32, RtOutput::DISTANCE_F32>();

    // Equivalent but using the output handler in a lidar. Using the
    // output handler verifies the size of your struct vs fields.

    auto pointAndDistanceOutput = lidar->getOutputHandler()->add<RayPointAndDistance,
                                                                 RtOutput::XYZ_VEC3_F32,
                                                                 RtOutput::DISTANCE_F32>();

    // perform raytrace
    auto view = pointAndDistanceOutput->view<RayPointAndDistance>();
    for ( const auto& pointAndDistance : view )
      ...
  */
  class AGXSENSOR_EXPORT RtOutput : public RtSystemNode, public ISensorOutput
  {
    public:
      // Types mirror the types defined by ROS2 sensor_msgs/PointField
      enum FieldElementType : uint8_t
      {
          EMPTY = 0,          
          I8,
          U8,
          I16,
          U16,
          I32,
          U32,
          F32,
          F64
      };

      enum Field : int32_t
      {
        /** 3x 32-bit floating point position of the hit. */
        XYZ_VEC3_F32 = 1,
        /** 1x 32-bit floating point intensity of the hit. */
        INTENSITY_F32,
        /** Placeholder - not calculated.  */
        INTENSITY_U8,
        /** 1x 32-bit flag (0 or 1) indicating if the hit is a hit or miss. */
        IS_HIT_I32,
        /** Placeholder - not calculated.  */
        IS_GROUND_I32,
        /** 1x 32-bit uint index of the ray. */
        RAY_IDX_U32,
        /** 1x 32-bit int index of the entity being hit. */
        ENTITY_ID_I32,
        /** 1x 32-bit floating point distance of hit from emission origin. */
        DISTANCE_F32,
        /** Placeholder - not calculated.  */
        AZIMUTH_F32,
        /** Placeholder - not calculated.  */
        ELEVATION_F32,
        /** Placeholder - not calculated.  */
        RING_ID_U16,
        /** Placeholder - not calculated.  */
        RETURN_TYPE_U8,
        /** 1x 64-bit floating point timestamp of the raytracing scene. */
        TIME_STAMP_F64,
        /** Placeholder - not calculated.  */
        TIME_STAMP_U32,
        /** Placeholder - not calculated.  */
        ABSOLUTE_VELOCITY_VEC3_F32,
        /** Placeholder - not calculated.  */
        RELATIVE_VELOCITY_VEC3_F32,
        /** Placeholder - not calculated.  */
        RADIAL_SPEED_F32,
        /** Placeholder - not calculated.  */
        POWER_F32,
        /** Placeholder - not calculated.  */
        RCS_F32,
        /** Placeholder - not calculated.  */
        NOISE_F32,
        /** Placeholder - not calculated.  */
        SNR_F32,
	    /** 3x 32-bit float normal vector of the entity surface where the hit-point is located. */
        NORMAL_VEC3_F32,
	    /** 
        1x 32-bit float incident angle of the ray hitting entity surface in radians.
	    In range [0, PI/2) rad, where 0 means the ray hit the surface perpendicularly.
	    */
        INCIDENT_ANGLE_F32,
	    /** 12x 32-bit float row major 3x4 matrix describing pose of the ray in the world coordinate system. */
        RAY_POSE_MAT3x4_F32,
        /** (Number of fields indicator) */
        NUM_FIELDS,
        /** 8-bit padding */
        PADDING_8 = 1024,
        /** 16-bit padding */
        PADDING_16,
        /** 32-bit padding */
        PADDING_32,
      };

      using FieldVector = std::vector<Field>;

      struct DataState
      {
        enum Flag : agx::UInt32
        {
          FETCHING_DATA = 1 << 0,
          SYNCHRONIZED = 1 << 1,
          HAS_UNREAD_DATA = 1 << 2, /**< Like the opposite of SYNCHRONIZED but is user controlled. */
        };
      };

      using State = agx::AtomicState<DataState::Flag>;

    public:
      /**
      \return unique hash of the given type T
      */
      template<typename T>
      static size_t getTypeHash();

      /**
      \return type name of the given type T
      */
      template<typename T>
      static const char* getTypeName();

      /**
      \param field - data field
      \return data size of the field
      */
      static size_t sizeOf( Field field );

      /**
      \param field - data field
      \return name of the given field
      */
      static const char* getFieldName( Field field );

      /**
      \param field - data field
      \return amount of elements in the field
      */
      static size_t getFieldElementCount(Field field);

      /**
      \param field - data field
      \return type of elements in the field
      */
      static FieldElementType getFieldElementType(Field field);

      /**
      \param field - data field
      \return names of individual elements in field (e.g. "x", "y", and "z" for position)
      */
      static const std::vector<std::string>& getFieldElementNames(Field field);

    public:
      /**
      Default constructor.
      */
      RtOutput();

      /**
      Add one or more fields, build a matching data structure.
      */
      template<Field... field>
      void build();

      /**
      Add new field.
      \param field - field to add
      \return this
      */
      RtOutput& add( Field field );

      /**
      "Element size" is the summed data size of the currently
      added fields.
      \return the current, summed, data size of the added fields
      */
      size_t getElementSize() const override;

      /**
      Synchronizes the data before returning if new data is available.
      \return the binary data buffer in the current state
      */
      const BinaryOutputBuffer& getData() override;

      /**
      Similar to view() but without data synchronization, this
      is the current data available.
      \return view of data as type T
      */
      template<typename T>
      BinaryOutputView<T> viewUnsafe() const;

#ifndef SWIG
      /**
      Unsafe data buffer access, use only this if you know the call is
      being made in context of raytrace and data synchronization. Use
      view or getSynchronizedData instead.
      */
      BinaryOutputBuffer& getDataUnsafe();
#endif

      /**
      Synchronizes the data before returning if new data is available.
      \return the binary data buffer in the current state
      */
      const BinaryOutputBuffer& getSynchronizedData();

      /**
      \param markAsRead - true to reset unread state, resulting in hasUnreadData returning
                          false until new data is available. False to not mark the current
                          data as read.
      \return true when the data has been updated but not read/used/fetched by
              the user, false if the data has been read
      */
      bool hasUnreadData( bool markAsRead = true ) override;

      /**
      \return true if this output contains the given field.
      */
      bool contains( Field field ) const;

      /**
      Offset in bytes of field in the pack of fields. Note that the total size of this
      output will be returned if \p field is not part of this output.
      \param field - field to get offset to
      \return offset in bytes to the given field
      */
      size_t getOffset( Field field ) const;

      /**      
      \return A const reference to the internal vector containg the fields of the output.
      */
      const std::vector<Field>& getFields() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~RtOutput() = default;

      virtual void preRaytrace();

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtOutput );

    protected:
      virtual RtNode* createNode() override;

    private:
      void fetchData();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      FieldVector m_fields;
      size_t m_elementSize;
      BinaryOutputBuffer m_output;
      State m_state;
  };

  template<typename T>
  size_t RtOutput::getTypeHash()
  {
    static const std::type_info& info = typeid( T );
    return info.hash_code();
  }

  template<typename T>
  const char* RtOutput::getTypeName()
  {
    static const std::type_info& info = typeid( T );
    return info.name();
  }

  template<RtOutput::Field... field>
  void RtOutput::build()
  {
    m_fields.clear();
    m_elementSize = 0u;
    ( this->add( field ), ... );
  }

  template<typename T>
  BinaryOutputView<T> RtOutput::viewUnsafe() const
  {
    if ( !RtOutput::verifySize<T>( this->getElementSize() ) )
      return {};
    return m_output.view<T>();
  }

  DOXYGEN_START_INTERNAL_BLOCK()
  class AGXSENSOR_EXPORT RtOutputVec4f : public RtOutput // swig
  {
  };
  DOXYGEN_END_INTERNAL_BLOCK()
}
