/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/OdometerOutput.h>
#include <agxSensor/MonoaxialGaussianNoise.h>
#include <agxSensor/MonoaxialSignalResolution.h>
#include <agxSensor/MonoaxialSignalScaling.h>
#include <agxSensor/SensorOutputHandler.h>

#include <optional>
#include <utility>

namespace agxSensor
{
  using OptionalUniqueId = std::optional<size_t>;
  using OptionalOdometerOutputMapping = std::optional<std::pair<size_t, OdometerOutput*>>;

  AGX_DECLARE_POINTER_TYPES( OdometerOutputHandler );

  /**
  Handler responsible for how the odometer readings are assembled and provided to the user.
  */
  class AGXSENSOR_EXPORT OdometerOutputHandler : public SystemNode, public ISensorOutputHandler
  {
    public:
      /**
      Default constructor.
      */
      OdometerOutputHandler();

      /**
      Add the given odometer output to the output handler mapped to the given unique id.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \param output - odometer output instance to add
      \return true if \p uniqueId > 0, \p uniqueId is unique and \p output != nullptr
      */
      bool add( size_t uniqueId, OdometerOutput* output );

      /**
      Add the given odometer output to the output handler. The added OdometerOutput can later be
      accessed from get() by using the returned uniqueId.
      \param output - odometer output instance to add
      \return the unique id assigned to the output on success, or nullopt on failure
      */
      OptionalUniqueId add( OdometerOutput* output );

      /**
      Build matching odometer output to a given data type T and add it mapped to the given unique id.
      This method verifies the size of the sum of data fields matches the size of T.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \return true if \p uniqueId > 0 and \p uniqueId is unique
      */
      template<typename T, OdometerOutput::Field... field>
      bool add( size_t uniqueId );

      /**
      Build matching odometer output to a given data type T. This method verifies the size of the
      sum of data fields matches the size of T. The created OdometerOutput returned from this method
      can later be accessed from get() by using the returned uniqueId.
      \return a pair containing the unique id assigned to the output and the built odometer output
      */
      template<typename T, OdometerOutput::Field... field>
      OptionalOdometerOutputMapping add();

      /**
      Access odometer output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      OdometerOutput* get( size_t uniqueId );

      /**
      Access odometer output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      ISensorOutput* getBase( size_t uniqueId ) override;

      /**
      Add a Gaussian noise instance to the system tree.
      \param noise - output noise to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( MonoaxialGaussianNoise* noise );

      /**
      Remove the Gaussian noise instance from the system tree. The noise will no longer be applied
      to the output data if removed successfully.
      \param noise - noise to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( MonoaxialGaussianNoise* noise );

      /**
      \param noise - noise instance to check if present and active
      \return true if \p noise is part of this output handler
      */
      bool contains( const MonoaxialGaussianNoise* noise ) const;

      /**
      Add a resolution binning instance to the system tree.
      \param resolution - output resolution binning to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( MonoaxialSignalResolution* resolution );

      /**
      Remove the resolution binning instance from the system tree. The noise will no longer be
      applied to the output data if removed successfully.
      \param resolution - resolution binning to remove
      \return true if removed successfully, false if nullptr or not part of this system tree
      */
      bool remove( MonoaxialSignalResolution* resolution );

      /**
      \param resolution - resolution binning to check if present and active
      \return true if \p resolution is part of this output handler
      */
      bool contains( const MonoaxialSignalResolution* resolution ) const;

      /**
      Add a signal scaling instance to the system tree.
      \param scaling - signal scaling to add to the system tree
      \return true if added successfully, false if nullptr or already added
      */
      bool add( MonoaxialSignalScaling* scaling );

      /**
      Remove the signal scaling instance from the system tree. The scaling will no longer be applied
      to the output data if removed successfully.
      \param scaling - scaling to remove
      \return true if removed successfully, false if nullptr or not part of this
              system tree
      */
      bool remove( MonoaxialSignalScaling* scaling );

      /**
      \param scaling - signal scaling instance to check if present and active
      \return true if \p scaling is part of this output handler
      */
      bool contains( const MonoaxialSignalScaling* scaling ) const;

      /**
      \return all added and active output modifiers
      */
      IMonoaxialSignalSystemNodeRefVector getOutputModifiers() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void synchronize( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::OdometerOutputHandler );

    protected:
      using OutputTable = agx::HashTable<size_t, OdometerOutputRef>;

    protected:
      bool addModifierInternal( IMonoaxialSignalSystemNode* node );
      bool removeModifierInternal( IMonoaxialSignalSystemNode* node );
      bool containsModifierInternal( const IMonoaxialSignalSystemNode* node ) const;

      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      OutputTable m_outputs;
      MonoaxialSignalSystemNodeRefVector m_outputModifiers;
      SystemNodeRef m_signalAssembler;
      MonoaxialSignalScalingRef m_wheelScaler;
      SystemNodeRef m_tailNode;
  };

  template<typename T, OdometerOutput::Field... field>
  bool OdometerOutputHandler::add( size_t uniqueId )
  {
    OdometerOutputRef output = new OdometerOutput();
    output->build<field...>();
    return add( uniqueId, output );
  }

  template<typename T, OdometerOutput::Field... field>
  OptionalOdometerOutputMapping OdometerOutputHandler::add()
  {
    OdometerOutputRef output = new OdometerOutput();
    output->build<field...>();

    auto uniqueId = add( output );
    if ( uniqueId )
      return std::make_pair( *uniqueId, output );
    else
      return std::nullopt;
  }
}
