/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxSensor/LidarRayAngleGaussianNoise.h>
#include <agxSensor/RaytraceSystemNodeProxy.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( LidarRayDistortionHandler );

  /**
  Ray noise handler is responsible for all distortion application to rays before raytrace.

  This handler can handle multiple distortions, which will be processed in the order they were
  added.
  */
  class AGXSENSOR_EXPORT LidarRayDistortionHandler : public RtSystemNode
  {
    public:
      /**
      Default constructor.
      */
      LidarRayDistortionHandler();

      /**
      Add distortion, angle noise etc, to the ray.
      Distortions are processed in the order they were added.
      \param distortion - distortion configuration to add to the lidar
      \return true if added successfully, false if nullptr or already added
      */
      bool add( LidarRayDistortion* distortion );

      /**
      Remove the given distortion configuration from the lidar.
      \param distortion - distortion configuration to remove
      \return true if removed successfully, false if nullptr or if the distortion was not 
              previously added to the lidar
      */
      bool remove(LidarRayDistortion* distortion);

      /**
      \param distortion - distortion configuration to check if present and active
      \return true if the given distortion configuration is part of this lidar
      */
      bool contains( const LidarRayDistortion* distortion) const;

      /**
      \return all added lidar ray distortions
      */
      LidarRayDistortionRefVector getDistortions() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual RtNode* createNode() final override;

      virtual void onParentConnect( RtSystemNode& parent ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarRayDistortionHandler );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      template<typename LidarRayModifier>
      struct LidarRayDistortionData
      {
        LidarRayModifier node;
        RtSystemNodeRef proxy;
      };

    private:
      using LidarRayDistortionDataVector = std::vector<LidarRayDistortionData<LidarRayDistortionRef>>;

    private:
      LidarRayDistortionDataVector m_distortions;
  };

}
