/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agxSDK/TerrainInstance.h>
#include <agxSDK/TerrainToolInstance.h>
#include <agx/HashSet.h>
#include <agx/Task.h>

namespace agxSDK
{
  DOXYGEN_START_INTERNAL_BLOCK()

  class Simulation;
  AGX_DECLARE_POINTER_TYPES(TerrainManager);

  /*!
  This class manages Terrains and Tools in the simulation.

  The TerrainManager both keeps tracks of all the terrains and tools in the simulation but also
  executes their step events (preCollide, pre, post, last) in the correct order.
  */
  class AGXPHYSICS_EXPORT TerrainManager : public agx::Referenced
  {
  public:
    /// Constructor
    TerrainManager(Simulation* simulation);

    /**
    Add terrain (agxTerrain::Terrain) instance to the manager.
    \param terrain - terrain to add
    \return true if added, false if nullptr or already present in this or another simulation
    */
    bool add(agxSDK::TerrainInstance* terrain);

    /**
    Remove terrain (agxTerrain::Terrain) instance from the manager.
    \param terrain - terrain to remove
    \return true if removed, false if nullptr or not part of this simulation
    */
    bool remove(agxSDK::TerrainInstance* terrain);

    /**
    Add TerrainToolInstance to the manager.
    \param tool - tool to add
    \return true if added, false if nullptr or already present in this or another simulation
    */
    bool add(agxSDK::TerrainToolInstance* tool);

    /**
    Remove TerrainToolInstance from the manager.
    \param tool - tool to remove
    \return true if removed, false if nullptr or not part of this simulation
    */
    bool remove(agxSDK::TerrainToolInstance* tool);

    /**
    Is the terrain contained in the simulation?
    \param terrain - The terrain to test
    \return true if the terrain is contained in the simulation.
    \note Might be linear complexity in all the objects of the same type in the simulation.
    */
    bool contains(const agxSDK::TerrainInstance* terrain) const;

    /**
    Is the tool contained in the simulation?
    \param tool - The tool to test
    \return true if the tool is contained in the simulation.
    \note Might be linear complexity in all the objects of the same type in the simulation.
    */
    bool contains(const agxSDK::TerrainToolInstance* tool) const;

    /**
    Pre collide callback.
    */
    void preCollide();

    /**
    Pre solver callback.
    */
    void pre();

    /**
    Post solver callback.
    */
    void post();

    /**
    Callback to be executed at the end of the time step
    */
    void last();

    /**
    Clean up TerrainManager, remove all the terrains and tools.
    Used by Simulation::cleanup.
    */
    void cleanUp();


    /**
    Find (linear search) and return named TerrainInstance
    \param name - name of the terrain to find
    \return pointer to the found RigidBody body, null if not found
    */
    agxSDK::TerrainInstance* getTerrain(const agx::Name& name);


    /**
    Find (linear search) and return a pointer to a TerrainInstance with the given uuid
    \param uuid - uuid of the requested terrain
    \return pointer to the found RigidBody, null if not found
    */
    agxSDK::TerrainInstance* getTerrain(const agx::Uuid& uuid);

    /**
    Find (linear search) and return named TerrainInstance
    \param name - name of the terrain to find
    \return pointer to the found RigidBody body, null if not found
    */
    const agxSDK::TerrainInstance* getTerrain(const agx::Name& name) const;


    /**
    Find (linear search) and return a pointer to a TerrainInstance with the given uuid
    \param uuid - uuid of the requested terrain
    \return pointer to the found RigidBody, null if not found
    */
    const agxSDK::TerrainInstance* getTerrain(const agx::Uuid& uuid) const;

    /**
    Find (linear search) and return named TerrainToolInstance
    \param name - name of the tool to find
    \return pointer to the found RigidBody body, null if not found
    */
    agxSDK::TerrainToolInstance* getTool(const agx::Name& name);


    /**
    Find (linear search) and return a pointer to a TerrainToolInstance with the given uuid
    \param uuid - uuid of the requested rigid body
    \return pointer to the found RigidBody, null if not found
    */
    agxSDK::TerrainToolInstance* getTool(const agx::Uuid& uuid) ;

    /**
    Find (linear search) and return named TerrainToolInstance
    \param name - name of the tool to find
    \return pointer to the found RigidBody body, null if not found
    */
    const agxSDK::TerrainToolInstance* getTool(const agx::Name& name) const;


    /**
    Find (linear search) and return a pointer to a TerrainToolInstance with the given uuid
    \param uuid - uuid of the requested rigid body
    \return pointer to the found RigidBody, null if not found
    */
    const agxSDK::TerrainToolInstance* getTool(const agx::Uuid& uuid) const;


    /**
    \return the number of current terrains in the simulation
    */
    agx::UInt getNumTerrains() const;

    /**
    \param i - index (i < getNumTerrains())
    \return terrain instance at given index \p i
    */
    agxSDK::TerrainInstance* getTerrain(agx::UInt i) const;

    /**
    \return a PtrVector of all the terrain instances in the manager.
    */
    agxSDK::TerrainInstancePtrVector getTerrains() const;

    /**
    \return a PtrVector of all the tool instances in the manager.
    */
    agxSDK::TerrainToolInstancePtrVector getTools() const;

    /**
    \return whether interactions between the given tool and terrain should be enabled or disabled. (Default is true).
    */
    bool getToolTerrainInteractionEnable(const agxSDK::TerrainToolInstance* tool, const agxSDK::TerrainInstance* terrain) const;

    /**
    Set whether interactions between the given tool and terrain should be enabled or disabled.
    */
    void setToolTerrainInteractionEnable(const agxSDK::TerrainToolInstance* tool, const agxSDK::TerrainInstance* terrain, bool enable);

    void store(agxStream::OutputArchive& out) const;
    void restore(agxStream::InputArchive& in);

  protected:
    /// Destructor
    virtual ~TerrainManager();

  private:
    Simulation* m_simulation;

    TerrainInstanceRefVector  m_terrains;
    TerrainToolInstanceRefVector   m_tools;
    agx::HashSet<agx::UInt32> m_toolTerrainInteractionDisableTable;
    agx::ParallelTaskRef      m_toolLastTask;

  };
  DOXYGEN_END_INTERNAL_BLOCK()
}
