/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <memory>

#include <agxCollide/Geometry.h>
#include <agxOpenPLX/OpenPlxToAgxMapper.h>
#include <agxOpenPLX/export.h>

namespace openplx
{
  namespace Physics3D
  {
    namespace Bodies
    {
      class RigidBody;
    }

    class System;
  }

  namespace Visuals
  {
    namespace Geometries
    {
      class Box;
      class Cylinder;
      class Sphere;
      class ExternalTriMeshGeometry;
      class ConvexMesh;
      class Geometry;
    }
  }
}

namespace agxopenplx
{

  struct AGXOPENPLX_EXPORT DebugFrameData
  {
      agx::RigidBodyRef body;
      agx::AffineMatrix4x4 localTransform;
      float scale;
  };

  /**
   * @brief Creates AGX visuals for OpenPLX objects.
   *
   */
  class OpenPlxToAgxVisualsMapper
  {
    public:
      AGXOPENPLX_EXPORT OpenPlxToAgxVisualsMapper(
        std::shared_ptr<OpenPlxToAgxMapper> openplx_to_agx_mapper,
        std::shared_ptr<openplx::ErrorReporter> error_reporter, bool debug_render_frames = false);
      AGXOPENPLX_EXPORT bool mapObject(std::shared_ptr<openplx::Core::Object> object);

      AGXOPENPLX_EXPORT agxSDK::Simulation* getSimulation() const;

      AGXOPENPLX_EXPORT const std::vector<agxCollide::GeometryRef>& getAddedVisualGeometries() const;
      AGXOPENPLX_EXPORT const std::vector<DebugFrameData>& getAddedVisualFrames() const;
      AGXOPENPLX_EXPORT const std::vector<agxTerrain::TerrainRef>& getAddedVisualTerrains() const;

      friend class OpenPlxSensorsMapper;

    private:
      void mapBox(const openplx::Visuals::Geometries::Box& box, const agxCollide::GeometryRef& agx_geometry);
      void mapCylinder(
        const openplx::Visuals::Geometries::Cylinder& cylinder, const agxCollide::GeometryRef& agx_geometry);
      void mapSphere(const openplx::Visuals::Geometries::Sphere& sphere, const agxCollide::GeometryRef& agx_geometry);
      void mapExternalTriMeshGeometry(
        const openplx::Visuals::Geometries::ExternalTriMeshGeometry& ext_trimesh_geometry,
        const agxCollide::GeometryRef& agx_geometry);
      void mapConvexMesh(
        const openplx::Visuals::Geometries::ConvexMesh& convex_mesh, const agxCollide::GeometryRef& agx_geometry);

      void mapGeometry(
        const openplx::Visuals::Geometries::Geometry& geometry, agxCollide::GeometryRef& agx_geometry);
      bool mapSystem(const std::shared_ptr<openplx::Physics3D::System>& system);
      bool mapBody(const std::shared_ptr<openplx::Physics3D::Bodies::RigidBody>& rigid_body);
      bool mapTerrain(const std::shared_ptr<openplx::Terrain::Terrain>& b_terrain);

      // Used for debug rendering
      void debugRenderMateConnectors(const std::shared_ptr<openplx::Physics3D::System>& system);
      void mapMate(const std::shared_ptr<openplx::Physics3D::Interactions::Mate>& mate);
      void mapMateConnector(const std::shared_ptr<openplx::Physics3D::Interactions::MateConnector>& mate_connector);
      void mapRange(
        const std::shared_ptr<openplx::Physics3D::Interactions::RangeInteraction1DOF>& range,
        agx::Angle::Type angle_type);

      void mapVisuals();
      void mapMaterials();

      void debugRenderFrame(agx::RigidBody* rigid_body, agx::FrameRef frame = nullptr, double scale = 1.0);
      void debugRenderRange(agx::RigidBody* rigidBody, agx::FrameRef frame, double start, double end);

      /**
       * @brief Given a rotational range, the convex spiral cake is a segment of a
       *        spiral that visualizes the range. One "cake" covers at maximum PI radians.
       *        The full range is then visualized by a number of these pieces of "cake".
       */
      agxCollide::GeometryRef generateConvexSpiralCake(double start, double end);

      std::shared_ptr<OpenPlxToAgxMapper> m_openplx_to_agx_mapper;
      std::shared_ptr<openplx::ErrorReporter> m_error_reporter;
      bool m_debug_render_frames;

      std::string m_source_id;
      openplx::Token m_root_token;

      std::unordered_set<std::shared_ptr<openplx::Physics3D::Interactions::MateConnector>> m_debug_rendered_mate_connectors;
      std::unordered_set<std::shared_ptr<openplx::Visuals::Geometries::Geometry>> m_mapped_geometries;
      std::unordered_map<std::shared_ptr<openplx::Visuals::Geometries::Geometry>, agxCollide::GeometryRef>
        m_geometry_map;
      std::unordered_set<std::shared_ptr<openplx::Physics3D::Bodies::Body>> m_body_set;

      // Lookup for visuals.
      std::vector<agxCollide::GeometryRef> m_mapped_visual_geometries;
      std::vector<DebugFrameData> m_mapped_visual_frames;
      std::vector<agxTerrain::TerrainRef> m_mapped_visual_terrains;
  };
}
