/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <memory>
#include <vector>
#include <agx/TimeStamp.h>
#include <openplx/Object.h>
#include <openplx/Robotics/Signals/RobotInput.h>
#include <openplx/Simulation/Simulation_all.h>
#include <agxOpenPLX/SignalQueue.h>
#include <click/Server.h>

namespace agxopenplx
{

  /**
   * @brief Propagates Click input signals to openplx
   * preFrame must be called continuously in main loop or similar.
   */
  class AGXOPENPLX_EXPORT ClickInputListener
  {

    public:
      ClickInputListener(
        std::shared_ptr<click::Server>& server, const std::shared_ptr<openplx::Core::Object>& openplx_scene,
        std::shared_ptr<InputSignalQueue> input_queue, double time_step, bool autostepping_enabled);
      virtual ~ClickInputListener() = default;

      virtual bool preFrame(const agx::TimeStamp&);
      /**
       * @brief Enable autostepping
       */
      void set_autostepping_enabled(bool enabled);
      /**
       * @brief Return whether autostepping is enabled
       */
      bool is_autostepping_enabled() const;
      bool is_click_signal_available() const;
      void setSensorRequestCallback(std::function<void(const agx::TimeStamp&)> callback);
      void updateScene(const std::shared_ptr<openplx::Core::Object>& openplx_scene);
      void updateInputQueue(const std::shared_ptr<InputSignalQueue>& input_queue);

      // Below callbacks are virtual methods instead of std::function so that they can easily access derived class
      // members
      /**
       * @brief called when autostep should be turned off
       */
      virtual void autostepOffCallback();

      /**
       * @brief called when autostep should be turned on
       */
      virtual void autostepOnCallback();

      /**
       * @brief called when the application should be stopped
       */
      virtual void stopApplicationCallback();

      /**
       * @brief called when a resetmessage is received, before sending a resetmessage back
       */
      virtual void resetCallback();

      /**
       * @brief returns the current OpenPLX scene
       */
      std::shared_ptr<openplx::Core::Object> getScene();

      /**
       * @brief returns true if ClickInputListener has the input_queue
       */
      bool hasQueue(const std::shared_ptr<InputSignalQueue>& input_queue);

    private:
      std::shared_ptr<click::Server> m_server;
      std::shared_ptr<openplx::Core::Object> m_openplx_scene;
      std::shared_ptr<InputSignalQueue> m_input_queue;
      bool m_autostepping_enabled;
      double m_time_step;
      bool m_click_signal_available;
      std::function<void(const agx::TimeStamp&)> m_sensor_request_callback;
  };
}
