/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxOSG/export.h>
#include <agxRender/RenderProxy.h>
#include <agxRender/RenderManager.h>

#include <agx/PushDisableWarnings.h> // Disabling warnings. Include agx/PopDisableWarnings.h below!
#include <osg/ref_ptr>
#include <osg/Group>
#include <osg/MatrixTransform>
#include <osg/Material>
#include <osg/AutoTransform>
#include <osg/Shape>
#include <osg/ShapeDrawable>
#include <osgText/Text>
#include <agx/PopDisableWarnings.h> // End of disabled warnings.

#include <agxOSG/GraphRenderer.h>
#include <agxOSG/RenderText.h>
#include <agxOSG/utils.h>
#include <agxOSG/PointSpriteDrawable.h>

#ifdef _MSC_VER
# pragma warning(push)
# pragma warning( disable : 4355 ) // warning C4355: 'this' : used in base member initializer list
#endif

namespace agxOSG
{

  class SphereProxy;
  class LineProxy;

  AGX_DECLARE_POINTER_TYPES(RenderProxyFactory);

  /**
  Implementation of the abstract class from the agxRender namespace, this class is responsible
  for creating RenderProxy of various types and render them as efficiently as possible in OpenSceneGraph.

  There are several node-roots:

  - getTextNode() returns the root of the Text, which is projected into 2D and
  rendered without lights.

  - getRootNode() - Parent of both: getSolidRootNode() and getWireFrameRootNode()
  */
  class AGXOSG_EXPORT RenderProxyFactory : public agxRender::RenderProxyFactory
  {
  public:
    /**
    Set new detail ratio for the rendered geometries.
    */
    static void  setCurrentDetailRatio(float detailRatio);

    /**
    \return the current detail ratio
    */
    static float getCurrentDetailRatio();

    /**
    \return the default detail ratio
    */
    static float getDefaultDetailRatio();

  public:
    RenderProxyFactory();

    /**
    \return the implementation of a GraphRenderer for osg.
    */
    agxRender::Graph::GraphRenderer* getGraphRenderer() override;


    /**
    \param radius - The radius of the new sphere
    \return a pointer to a new SphereProxy with specified radius
    */
    agxRender::SphereProxy* createSphere(float radius) override;

    /**
    \param shape - The sphere that should be rendered
    \return a pointer to a new SphereProxy with specified radius
    */
    agxRender::SphereProxy* createSphere(agxCollide::Sphere *shape) override;

    /**
    \param text - The text
    \param pos - Position of the text. Currently only x,y is used.
    \param size - Size of text. Used to scale the actual text used.
    \return a pointer to a new TextProxy
    */
    agxRender::TextProxy* createText(const agx::String& text, const agx::Vec3& pos, float size = 1) override;

    /**
    \param p1, p2 - Start, end points in WORLD coordinate system
    \return a pointer to a new LineProxy
    */
    agxRender::LineProxy* createLine(const agx::Vec3& p1, const agx::Vec3& p2) override;

    /**
    \param shape - Line that should be rendered
    \return a pointer to a new LineProxy
    */
    agxRender::LineProxy* createLine(agxCollide::Line *shape) override;

    /**
    \param radius - The radius of a cylinder
    \param height - The height of a cylinder
    \return a pointer to a new CylinderProxy
    */
    agxRender::CylinderProxy* createCylinder(float radius, float height) override;

    /**
    \param shape - Cylinder that should be rendered
    \return a pointer to a new CylinderProxy
    */
    agxRender::CylinderProxy* createCylinder(agxCollide::Cylinder *shape) override;

    /**
    \param outerRadius - The outer radius for the a cylinder
    \param height - The height for the cylinder
    \param thickness - The thickness for the cylinder
    \return a pointer to a new HollowCylinderProxy
    */
    agxRender::HollowCylinderProxy* createHollowCylinder(float outerRadius, float height, float thickness) override;

    /**
    \param shape - HollowCylinder that should be rendered
    \return a pointer to a new HollowCylinderProxy
    */
    agxRender::HollowCylinderProxy* createHollowCylinder(agxCollide::HollowCylinder* shape) override;

    /**
    \param radius - The radius of a capsule
    \param height - The height of a capsule
    \return a pointer to a new CapsuleProxy
    */
    agxRender::CapsuleProxy* createCapsule(float radius, float height) override;

    /**
    \param shape - Capsule that should be renderered
    \return a pointer to a new CapsuleProxy
    */
    agxRender::CapsuleProxy* createCapsule(agxCollide::Capsule* shape) override;

    /**
    \param radius - The radius of a capsule
    \param height - The height of a capsule
    \param previousEndPoint0 - The previous lower end point of the WireShape.
    \param previousEndPoint1 - The previous upper end point of the WireShape.
    \return a pointer to a new WireShapeProxy
    */
    agxRender::WireShapeProxy* createWireShape(float radius, float height,
      const agx::Vec3& previousEndPoint0, const agx::Vec3& previousEndPoint1) override;

    /**
    \param halfExtents - The size of the box
    \return a pointer to a new BoxProxy with specified size
    */
    agxRender::BoxProxy* createBox(const agx::Vec3& halfExtents) override;

    /**
    \param shape - Box that should be rendered
    \return a pointer to a new BoxProxy
    */
    agxRender::BoxProxy* createBox(agxCollide::Box* shape) override;

    /**
    Create and return a RenderProxy for a Heightfield.
    \param shape - The heightfield for which a HeightfieldProxy will be created.
    */
    agxRender::HeightFieldProxy* createHeightfield(agxCollide::HeightField* shape) override;

    /**
    Create and return a RenderProxy for a Trimesh.
    \param shape - The mesh shape for which a TrimeshProxy will be created.
    */
    agxRender::TrimeshProxy* createTrimesh(agxCollide::Trimesh* shape) override;

    /**
    \param radius - The radius of the cone
    \param height - The height of the cone
    \return a pointer to a new ConeProxy
    */
    agxRender::ConeProxy* createCone(float radius, float height) override;

    /**
    \param shape - Cone that should be rendered
    \return a pointer to a new TruncatedConeProxy
    */
    agxRender::TruncatedConeProxy* createCone(agxCollide::Cone* shape) override;

    /**
    \param topRadius - The top radius of the cone
    \param bottomRadius - The bottom radius of the cone
    \param height - The height of the cone
    \return a pointer to a new ConeProxy
    */
    agxRender::TruncatedConeProxy* createTruncatedCone(float topRadius, float bottomRadius, float height) override;

    /**
    \param topRadius - The top radius of the cone
    \param bottomRadius - The bottom radius of the cone
    \param height - The height of the cone
    \param thickness - The thickness of the cone
    \return a pointer to a new HollowConeProxy
    */
    agxRender::HollowConeProxy* createHollowCone(float topRadius, float bottomRadius, float height, float thickness) override;

    /**
    \param shape - HollowCone that should be rendered
    \return a pointer to a new HollowConeProxy
    */
    agxRender::HollowConeProxy* createHollowCone(agxCollide::HollowCone *shape) override;

    /**
    \param normal - The normal of a plane
    \param distance - The scalar part of the plane
    \return a pointer to a new PlaneProxy
    */
    agxRender::PlaneProxy* createPlane(const agx::Vec3& normal, agx::Real distance) override;

    /**
    \param shape - The plane that should be rendered
    \return a pointer to a new PlaneProxy
    */
    agxRender::PlaneProxy* createPlane(agxCollide::Plane* shape) override;

    /**
    \param contacts - Vector with all contacts that should be visualized
    \return a pointer to a new ContactsProxy
    */
    agxRender::ContactsProxy* createContacts(const agxCollide::GeometryContactPtrVector& contacts) override;

    agxRender::RigidBodyBatchRenderProxy* createRigidBodies(const agx::RigidBodyPtrSetVector* enabledBodies) override;

    agxRender::WireRenderProxy* createWire(float radius, const agx::Vec3& color) override;


    void setSphereBatchRenderModeSprites(bool flag);
    agxRender::RenderProxy* createSphereBatchRenderer(
      agxData::Buffer* positions, agxData::Buffer* rotations, agxData::Buffer* radii, agxData::Buffer* colors,
      agxData::Buffer* enableRendering, agxData::Value* bound, agx::Component* context) override;

    /**
    The text node is not child of the node returned in getRootNode(), hence this node MUST
    be added separately.
    \return the node under which all the Text element will go
    */
    osg::Group* getTextNode() { return m_textParent.get(); }

    /**

    \return the node under which the solid and wire frame nodes lies.
    */
    osg::Node* getRootNode() { return m_parent; }

    /**
    \return the parent of all solid renderable objects
    */
    osg::Group* getSolidRootNode() { return m_solidParent; }

    /**
    \return the parent of all wireframe renderable objects
    */
    osg::Group* getWireframeRootNode() { return m_wireframeParent; }

    /**
    Set a transformation which will be applied to all transformations for RenderProxy's.
    This can be used to transform all objects to the center of the earth instead of trying to render on the surface of the earth (6000km).
    This transform is by default I.
    If you for example do m.setTranslate(-10,0,0) all RenderProxy's will be rendered translated -10 units in X.
    If a RenderManager is present, RenderManager::update() will be called
    */
    void setGlobalTransform(const agx::AffineMatrix4x4& m) override;

    /**
    \return the current global transformation
    */
    const agx::AffineMatrix4x4& getGlobalTransform() const { return m_invTransform; }

    /**
    Set the RenderMode for a specified node
    */
    void setRenderMode(agxRender::RenderProxy* proxy, osg::Node* node, agxRender::RenderProxy::RenderMode mode);


    static osg::Geode* createGeode(osg::Shape* shape, osg::ShapeDrawable** shapeDrawable = nullptr, float detailRatio = RenderProxyFactory::getDefaultDetailRatio());

  public:
    /// Class for storing OSG specific data for each RenderProxy
    template<typename T>
    struct ShapeData {
      ShapeData() : material(nullptr) {}
      ShapeData(const ShapeData& other)
      {
        shape = other.shape;
        geode = other.geode;
        material = other.material;
        stateSet = other.stateSet;
      }

      osg::ref_ptr<T> shape;
      osg::ref_ptr<osg::Geode> geode;
      osg::Material* material;
      osg::ref_ptr<osg::StateSet> stateSet;
    };


    /**
    Add a child to the root, default getDefaultRenderMode() will be used for determining if
    the node should go into wireframe and or solid parent.
    */
    void addChild(osg::Node* node, agxRender::RenderProxy::RenderMode mode);
    void addChild(osg::Node* node, agxRender::RenderProxy* proxy);

  protected:
    /// Destructor
    virtual ~RenderProxyFactory();

    osg::ref_ptr<osg::Group> m_solidParent;
    osg::ref_ptr<osg::Group> m_wireframeParent;
    osg::ref_ptr<osg::Group> m_parent;
    osg::ref_ptr<osg::Group> m_textParent;
    agx::ref_ptr<agxOSG::GraphRenderer> m_graphRenderer;
    agx::AffineMatrix4x4 m_invTransform;
    bool m_sphereBatchRenderModeSprites;
  };

}
