/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/config/AGX_USE_WEBSOCKETS.h>
#if AGX_USE_WEBSOCKETS()
#include <agx/agxCore_export.h>

#include <civetweb/CivetServer.h>
#include <memory>
#include <thread>
#include <atomic>
#include <mutex>
#include <string>
#include <chrono>
#include <agx/String.h>

#ifdef _MSC_VER
  #pragma warning(push)
  //  warning C4251:  'X' : class 'Y' needs to have dll-interface to be used by clients of class 'Z'
  #pragma warning(disable : 4251)
#endif

namespace agxNet
{
  namespace agxWebServer
  {
    /**
     Thin wrapper around CivetServer acting as WebServer for the WebDebugger interface.
     - Serves files from the current working directory
     - Listens on 0.0.0.0:<port>
     - Ensures correct MIME types for .js, .mjs, .ts, and .wasm
     - Runs in a separate thread
     */
    class AGXCORE_EXPORT WebDebuggerServer
    {
      public:
        // Configuration constants
        inline static constexpr std::chrono::milliseconds STARTUP_TIMEOUT {2000};
        inline static constexpr std::chrono::milliseconds POLL_INTERVAL {10};
        inline static constexpr int MAX_PORT_RETRY_ATTEMPTS = 10;
        inline static constexpr int DEFAULT_PORT = 5173;
        /**
        Construct the server but do not start it yet.
        \param port - Port to serve (0 for auto-assign)
        \param directory - From which directory the server should serve. If left empty, directory will be the default package directory.
        */
        explicit WebDebuggerServer(int port = DEFAULT_PORT, const std::string& server_directory = "") noexcept;

        /**
        Destructor - stops the server if running
        */
        ~WebDebuggerServer() noexcept;

        // Non-copyable, non-movable (manages thread and resources)
        WebDebuggerServer(const WebDebuggerServer&) = delete;
        WebDebuggerServer& operator=(const WebDebuggerServer&) = delete;
        WebDebuggerServer(WebDebuggerServer&&) = delete;
        WebDebuggerServer& operator=(WebDebuggerServer&&) = delete;

        /**
        Start the HTTP server in a separate thread.
        Blocks until server is running and port is assigned.

        \throws CivetException on failure
        */
        void start();

        /**
        Stop the HTTP server and signal thread to exit.
        Safe to call multiple times.
        Does not block - use join() to wait for completion.
        */
        void stop() noexcept;

        /**
         Wait for the server thread to finish.
         Call after stop() to ensure clean shutdown.
         Safe to call multiple times.
         */
        void join() noexcept;

        /**
        \return True if the server is currently running
        */
        bool isRunning() const noexcept;

        /**
        \return Get the port actually used by the server (useful if 0 was passed).
                Returns 0 if server hasn't started yet.
        */
        int getPort() const noexcept;

        /**
        Get the local IP address for LAN access.
        \return Local IP address string, or "your-ip" if unavailable
        */
        static std::string getLocalIP() noexcept;

      private:
        void serverThreadFunc() noexcept;

        int m_port;
        std::unique_ptr<CivetServer> m_server;
        std::thread m_serverThread;
        std::atomic<bool> m_running;
        std::atomic<bool> m_stopRequested;
        agx::String m_serverDirectory;
        mutable std::mutex m_mutex;
    };
  }
}
  #ifdef _MSC_VER
  #pragma warning(pop)
#endif

#endif