#pragma once

#include <filesystem>
#include <optional>
#include <unordered_set>
#include <openplx/BundleConfig.h>
#include <openplx/ErrorReporter.h>
#include <openplx/Logger.h>

namespace openplx::Bundle
{
    class FindBundle
    {
        public:
            /**
             * Load one specific bundle, the first config.openplx found from starting_point.
             * If starting_point is a config.openplx, it will be used.
             * Otherwise, starting in directory of starting_point search upwards until root
             *
             * @param starting_point the file or directory to start from
             * @return std::optional<BundleConfig>
             */
            static std::optional<BundleConfig> findBundleFrom(const std::filesystem::path& starting_point, const std::shared_ptr<Logger>& logger);

            /**
             * @brief Search for a bundle with the given name in each path of paths
             * Search in all directories in paths after a config.openplx with the given name.
             * If not found there, continue searching in all direct subdirectories of each path in paths.
             * I.e does not look further down than one level below paths
             *
             * @param name the bundle name to search for
             * @param paths a vector with each entry being a directory path to search in
             * @return std::optional<BundleConfig>
             */
            static std::optional<BundleConfig> findBundle(std::string name, const std::vector<std::string>& paths, const std::shared_ptr<Logger>& logger);

            /**
             * @brief Load all bundles in paths
             * Search in all directories in paths after a config.openplx with the given name.
             * If not found there, continue searching in all direct subdirectories of each path in paths.
             * I.e does not look further down than one level below paths
             *
             * @param paths a vector with each entry being a directory path to search in
             * @return std::vector<BundleConfig>
             */
            static std::vector<BundleConfig> resolveBundleConfigFromPaths(const std::vector<std::string>& paths, const std::shared_ptr<Logger>& logger);

            /**
             * Parses a dependency string of the form <bundle name>==<version> into a pair { <bundle name>, <version> }
             *
             * @param dependency_string The string of the form, for example: Math==1.2.0
             * @return A pair, for example { "Math", "1.2.0" }
             */
            static std::pair<std::string, std::string> parseDependencyString(const std::string& dependency_string);

            /**
             * Resolves dependencies by extracting version information and expanding transient dependencies.
             *
             * @params bundle_config The unresolved bundle_config, resolved_dependencies will be populated
             * @param paths a vector with each entry being a directory path to search in
             * @param error_reporter The error reported where DependencyNotSatisfied errors will be reported
             * @returns true if the dependencies was resolved succesfully, false otherwise (an error should also be reported)
             */
            static bool resolveDependencies(BundleConfig& bundle_config, const std::vector<std::string>& paths, const std::shared_ptr<ErrorReporter>& error_reporter, const std::shared_ptr<Logger>& logger);
    };
}
