
#pragma once

#include <string>
#include <vector>
#include <memory>
#include <map>
#include <openplx/BundleConfig.h>
#include <openplx/AnalysisContext.h>
#include <openplx/Document.h>
#include <openplx/ModelDeclaration.h>
#include <openplx/MethodDeclaration.h>
#include <openplx/IndentedStream.h>
#include <openplx/Type.h>

namespace openplx {
    class Logger;
}

namespace openplx::Core
{
    class BundleGenerator
    {
        public:
            BundleGenerator(const std::string bundle_name,
                            std::vector<std::string> dependencies,
                            std::vector<ModelDeclPtr> model_declarations,
                            std::vector<DocPtr> documents,
                            std::shared_ptr<Logger> logger);

            void writeForwardDeclarationsTo(std::ostream &output_stream);
            void writeModelDeclarationTo(const std::string& bundle_name, ModelDeclaration& model, std::ostream &output_stream);
            void writeTraitDeclarationTo(const std::string& bundle_name, ModelDeclaration& model, std::ostream &output_stream);

            void writeBundleDeclarationTo(std::ostream &output_stream);
            void writeBundleDefinitionTo(std::ostream &output_stream);

            void writePythonSwigInterfaceTo(std::ostream& output_stream);
            void writePythonSwigModuleInterfaceTo(std::ostream& output_stream);
            void writeCsharpSwigInterfaceTo(std::ostream& output_stream);
            void writeCsharpModuleSwigInterfaceTo(std::ostream& output_stream);

            void writeDocumentMarkdownTo(
                const std::string& version,
                const std::string& bundle_name,
                const std::string& filename,
                const std::string& basename,
                const std::string& relative_path,
                const std::string& openplx_source,
                std::ostream& output_stream);
            void writeIndexMarkdownTo(
                const std::string& version,
                const std::string& title,
                const std::string& bundle_name,
                const std::string& relative_path,
                const std::string& index_source,
                std::ostream& os);

        private:
            std::string typeToCppType(const TypePtr& type, bool optional = false);
            void writeModelDefinitionTo(ModelDeclaration& model, std::ostream &output_stream);
            void writeTraitDefinitionTo(ModelDeclaration& trait, std::ostream &output_stream);
            void writeDocstringTo(is::IndentedStream& os, const std::vector<AnnotationPtr>& annotations);
            void writeFactoryTo(ModelDeclaration& model, std::ostream &output_stream);
            void writeMethodWrapperTo(
                    ModelDeclaration& model,
                    const MethodDeclPtr& method,
                    std::ostream &outputStream);
            void writeUnpackedAnyTo(is::IndentedStream& os, const std::string& var_name, const TypePtr& var_type);
            void writeUnpackedArrayArgsTo(is::IndentedStream& os, const MethodDeclPtr& method);
            void writeMethodArgsTo(
                is::IndentedStream& os,
                const MethodDeclPtr& method);
            void writeOperatorOverloadSignatureTo(is::IndentedStream& os, const OpOverloadPtr& op_overload, bool skip_return_type = false);
            void writeOperatorOverloadWrapperNameTo(is::IndentedStream& os, const OpOverloadPtr& op_overload);
            void generateExtractObjectFieldsToBody(is::IndentedStream& os, ModelDeclaration& model);
            void generateExtractEntriesToBody(is::IndentedStream &os, ModelDeclaration &model);
            void generateTriggerOnInitToBody(is::IndentedStream& os, ModelDeclaration& model);
            void generateSetDynamicBody(is::IndentedStream& os, ModelDeclaration& model);
            void generateGetDynamicBody(is::IndentedStream& os, ModelDeclaration& model);
            std::vector<ModelDeclPtr> calculateModelDependencies(ModelDeclPtr model);
            std::string m_bundle_name;
            std::vector<std::string> m_bundle_dependencies;
            std::vector<ModelDeclPtr> m_all_models;
            std::vector<DocPtr> m_bundle_documents;
            std::shared_ptr<Logger> m_logger;
    };
}
