/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/DeformerCollection.h>
#include <agx/Plane.h>
#include <agxTerrain/AggregateContactUtils.h>
#include <agxTerrain/TerrainFrictionModels.h>
#include <array>

namespace agxTerrain
{
  class Terrain;
  class Shovel;

  AGX_DECLARE_POINTER_TYPES(DeformController);

  class AGXTERRAIN_EXPORT DeformController : public agx::Referenced
  {
  public:
    DeformController( agx::Frame* parentFrame);

    void addDeformer(DeformerCollection* deformer);

    void clearDeformers();

    void addNotification(agxSDK::Simulation* simulation);

    void removeNotification(agxSDK::Simulation* simulation);

    void onPreCollide(Terrain* terrain, bool shouldDeformMass);

    void onPre(Terrain* terrain, Shovel* shovel);

    void onPost( Shovel* shovel, Terrain* terrain);

    DeformerCollection* getDeformerCollection(size_t i);

    size_t getNumDeformers();

    ActiveZonePtrVector getActiveZones();

    agxCollide::GeometryPtrVector getActiveZoneGeometries();

    SoilParticleAggregatePtrVector getSoilParticleAggregates();

    agxCollide::GeometryContactPtrVector getGeometryContacts(agxSDK::Simulation *simulation) const;

    void onEnableChanged(bool enable);

    void initializeContactMaterials(Terrain* terrain, Shovel* shovel);

    void initializeContactMaterialsExternalDeformer(agx::RigidBody* parentBody, Terrain* terrain);

    void initializeAggregateTerrainContactMaterial(Terrain* terrain, size_t deformerMode);

    void initializeFrictionModels();

    void initializeDepthModels();

    void resetAggregateContactHistories();

    void resetAggregateContactHistory(agx::UInt i);

    const agxTerrain::AggregateContactDepthModel& getAggregateDepthModel(agx::UInt i) const;

    void setEnableCustomFrictionModels( bool enable );

    bool getEnableCustomFrictionModels() const;

    agx::Real calculateShovelAggregateContactArea(const agx::Vec3Vector& contactPoints, agx::Real voxelArea) const;

    /**
    Get the internal shovel <-> aggregate contact material of the specified deformer
    denoted by an index.
    \param i - index of the deformer which contact material should be accessed. [0-2]
    \return the contact material of the specified deformer.
    */
    agx::ContactMaterial* getAggregateShovelContactMaterial( agx::UInt i ) const;

    /**
    Get the internal aggregate <-> terrain contact material of the specified deformer
    denoted by an index.
    \param i - index of the deformer which contact material should be accessed. [0-2]
    \return the contact material of the specified deformer.
    */
    agx::ContactMaterial* getAggregateTerrainContactMaterial( agx::UInt i ) const;

  protected:
    virtual ~DeformController();

  private:
    enum DeformSide
    {
      BACK = 0,
      LEFT = 1,
      RIGHT = 2
    };

  private:

    void deformSolidMass(Terrain* terrain);

    bool shouldDeformMass(Terrain* terrain, Shovel* shovel);

    void createAggregateContactPoints(Terrain* terrain, Shovel* shovel);

    void createShovelContacts(Terrain* terrain, Shovel* shovel);

    void createTerrainContacts(Terrain* terrain, Shovel* shovel);

    agx::Vec3Vector findShovelTerrainIntersectionPoints( Terrain* terrain,
                                                         DeformerCollection* deformer );

    agx::Vec3Vector findAggregateTerrainIntersectionPoints( const agx::Line& wedgeCuttingEdge,
                                                            const agx::Line& wedgeFailureEdge );

    agx::Vec3 calculateAggregateNormal( const agx::Line& wedgeCuttingEdge,
                                        const agx::Line& wedgeFailureEdge ) const;

    void pushSoilToSides(Terrain* terrain,
      const agx::Vec3& fromVoxelWorldPos,
      const agx::Real& elementSize,
      const agx::Vec3& topVectorNormalized,
      const agx::Vec3& deformerNormal,
      const agx::Real& timeStep,
      const agx::Real& voxelSpeed,
      const agx::Vec2i& fromTerrainIndex,
      const float& occupancyInVoxel,
      const float& dynamicMassCompaction,
      const agx::Vec3i& voxel,
      agx::Vec3* voxelSideVelocity);

    void updateFrictionModels( Terrain* terrain );

    void DEBUG_DRAW();

  private:
    agx::Frame*                               m_parentFrame;
    agx::Vector<DeformerCollectionRef>        m_deformers;
    agx::Vector<AggregateContactDepthModel>   m_depthModels;
    agx::Vector<ShovelAggregateFrictionModelRef> m_frictionModels;
    agxCollide::LocalGeometryContactVector    m_contacts;
    agx::Vector<agx::ContactMaterialRef>      m_aggregateTerrainContactMaterials;
    agx::Vector<agx::ContactMaterialRef>      m_aggregateShovelContactMaterials;
    bool m_enableCustomFrictionModels;
  };
}