/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/TriaxialSignalNoiseNode.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( TriaxialSpectralGaussianNoise );

  /**
  Applies Gaussian noise to the triaxial signal based on a specified noise spectral density and
  the sample frequency of the signal.
  */
  class AGXSENSOR_EXPORT TriaxialSpectralGaussianNoise : public ITriaxialSignalNoiseNode
  {
    public:
      /**
      Construct this sample frequency dependent Gaussian noise based on the given \p noiseDensity.
      \param noiseDensity - sensor noise spectral density to configure the noise with in
                            (measurement unit)/sqrt(Hz) RMS
      \param seed - seed used to configure the pseudo-random perturbation generation
      */
      TriaxialSpectralGaussianNoise( const agx::Vec3& noiseDensity = agx::Vec3( 2.943 * 1e-3 ),
                                     agx::UInt32 seed = 3565u );

      /**
      Set the sensor noise spectral density.
      \param noiseDensity - Sensor noise spectral density to configure the noise with in
                            (measurement unit)/sqrt(Hz) RMS
      */
      void setNoiseDensity( const agx::Vec3& noiseDensity );

      /**
      \return noise spectral density in (measurement unit)/sqrt(Hz) RMS.
      */
      const agx::Vec3& getNoiseDensity() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( SystemNodeProxy& proxy, const CallbackData& data ) override;
      virtual void cleanup( SystemNodeProxy& proxy ) override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::TriaxialSpectralGaussianNoise );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Vec3 m_noiseDensity;
  };
}
