/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SensorOutput.h>

namespace agxSensor
{
  /**
  Common interface for sensor output handlers.
  */
  class AGXSENSOR_EXPORT ISensorOutputHandler
  {
    public:
      /**
      Default constructor.
      */
      ISensorOutputHandler() = default;

      /**
      Access sensor output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      ISensorOutput* get( size_t uniqueId );

      /**
      Access sensor output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      virtual ISensorOutput* getBase( size_t uniqueId ) = 0;

      /**
      Directly access an output view of the sensor output instance given the unique id from when it
      was added.
      \param uniqueId - unique id for the output instance
      \return output view of the sensor output if an output with the given \p uniqueId exists and if
              the view type aligns with the data, otherwise an empty output view is returned
      */
      template<typename T>
      BinaryOutputView<T> view( size_t uniqueId );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual ~ISensorOutputHandler() = default;

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  template<typename T>
  BinaryOutputView<T> ISensorOutputHandler::view( size_t uniqueId )
  {
    auto output = this->getBase( uniqueId );
    if ( output == nullptr ) {
      LOGGER_WARNING() << "Sensor output of unique id \"" << uniqueId << "\" hasn't been "
                          "registered. An empty view is returned." << LOGGER_ENDL();
      return {};
    }
    return output->template view<T>();
  }
}
