/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/RaytraceSystemNode.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( IRtSystemNode );

  /**
  Interface as RtSystemNode for model specific implementations that is part
  of the system tree of lidar instances. IRtSystemNode::createProxy is called
  when a lidar is built from a model, so implementations of this class will
  receive calls from every instance of the lidar.
  */
  class AGXSENSOR_EXPORT IRtSystemNode : public agx::Referenced, public agxStream::Serializable
  {
    public:
      using CallbackData = SystemNode::CallbackData;

    public:
      /**
      \return new, unique, system node (proxy) of this implementation
      */
      virtual RtSystemNodeRef createProxy();
      
      DOXYGEN_START_INTERNAL_BLOCK()
      // Interface like a RtSystemNode.
    public:
      virtual RtNode* createNode() = 0;
      virtual void synchronize( RtNode* node, const CallbackData& data );
      virtual void execute( RtNode* node, const CallbackData& data );
      virtual void cleanup( RtNode* node );

      AGXSTREAM_DECLARE_ABSTRACT_SERIALIZABLE( agxSensor::IRtSystemNode );

      DOXYGEN_END_INTERNAL_BLOCK()
  };

  AGX_DECLARE_POINTER_TYPES( RtSystemNodeProxy );

  /**
  Raytrace system node acting as a clone of model specific implementations. Instead
  of cloning actual nodes, this proxy node has an implementation instance with similar
  interface as RtSystemNode and forwards calls to the implementation. E.g., a ray
  pattern of a lidar model is a single instance that we would like to use in several
  lidar instances and any modification should instantly be reflected in all lidar
  instaces of that model.
  */
  class AGXSENSOR_EXPORT RtSystemNodeProxy : public RtSystemNode
  {
    public:
      /**
      Construct given an implementation of a raytrace system node.
      \param implementation - implementation instance != nullptr
      */
      RtSystemNodeProxy( IRtSystemNode* implementation );

      /**
      \return true if this node implements the given type, otherwise false
      */
      template<typename T>
      bool implements() const;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual RtNode* createNode() final override;
      virtual void synchronize( const CallbackData& data ) final override;
      virtual void execute( const CallbackData& data ) final override;
      virtual void cleanup() final override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtSystemNodeProxy );

    private:
      // Used during restore.
      RtSystemNodeProxy();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      IRtSystemNodeRef m_implementation;
  };

  template<typename T>
  bool RtSystemNodeProxy::implements() const
  {
    return m_implementation->is<T>();
  }

  template<typename T>
  RtSystemNode* RtSystemNode::findChildProxyFor() const
  {
    return findChildProxyFor<T>( this );
  }

  template<typename T>
  RtSystemNode* findChildProxyFor( const SystemNode* root )
  {
    if ( root == nullptr )
      return nullptr;

    SystemNode* node = root->findChild( []( const SystemNode& child )
                                        {
                                          return child.is<RtSystemNodeProxy>() &&
                                                 child.as<RtSystemNodeProxy>()->implements<T>();
                                        } );

    return node != nullptr ? node->as<RtSystemNode>() : nullptr;
  }
}
