/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/SystemNode.h>
#include <agxSensor/EnvironmentInternalData.h>
#include <agxSensor/MagneticField.h>
#include <agxSensor/RaytraceHandles.h>

#include <agxSDK/ISensorEnvironment.h>

#include <agx/ReferencedHandler.h>
#include <agx/AtomicState.h>
#include <agx/Timer.h>

#include <thread>

namespace agxCollide
{
  class Shape;
  class Geometry;
}

namespace agx
{
  class RigidBody;
}

namespace agxSDK
{
  class Simulation;
  class LinkedStructure;
}

namespace agxWire
{
  class Wire;
}

namespace agxTerrain
{
  class Terrain;
  class TerrainPager;
}

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Environment );

  /**
  Sensor environment implementation where different sensors collects data given a set
  of added objects, such as rigid bodies, geometries, shapes, wires, cables, terrains,
  etc.

  This sensor environment can be run in cooperative mode with an agxSDK::Simulation
  instance (normal use case) or standalone and is always integrated in its own thread.
  
  In cooperative mode, the stepping/integration of the sensor environment starts after the
  simulation step event 'preCollide'. New/updated sensor data is available from the simulation
  step event 'post'. This means that any changes, such as updated transforms or configuration
  changes, has to be made before calling simulation 'stepForward' or in a 'preCollide' step event.

  Important: When the sensor environment is integrated in its own thread it's undefined
             to interact with sensors and/or adding removing objects during the collision
             detection, contact event callbacks, pre step events and dynamics solvers
             integrations. Data access and object/sensor modifications are safe to do
             in post step events, post contact events, last step events and outside of
             'stepForward'.
  
  To create or access a sensor simulation using agxSDK::Simulation, do:

      // Cooperative:
      agxSensor::Environment* sensorEnv = agxSensor::Environment::getOrCreate( simulation );
      simulation->stepForward(); // Will step the sensor environment.

      // OR (also cooperative)
      agxSensor::EnvironmentRef sensorEnv = new agxSensor::Environment();
      simulation->setSensorEnvironment( sensorEnv );
      simulation->stepForward(); // Will step the sensor environment.

      // Standalone:
      agxSensor::EnvironmentRef sensorEnv = new agxSensor::Environment();
      sensorEnv->addNotification(); // Important, will create a separate thread that the
                                    // sensor environment is integrated in.
      while ( !done ) {
        sensorEnv->step( 0.0166667 ); // Read data from added objects and start the sensor environment
                                      // simulation step - in the thread created in addNotification.
        ... // Do other things.
        sensorEnv->fetchStepResults(); // Blocking until the sensor environment has been integrated and
                                       // new sensor output data is available.
      }
  */
  class AGXSENSOR_EXPORT Environment : public agxSDK::ISensorEnvironment
  {
    public:
      /**
      Returns an existing or creates and initializes a new instance of a sensor
      environment given a \p simulation. This method will only return nullptr if
      the given \p simulation is nullptr.
      \param simulation - simulation instance for the sensor environment
      \return sensor environment for the given simulation instance, nullptr if \p simulation is nullptr
      */
      static Environment* getOrCreate( agxSDK::Simulation* simulation );

      /**
      \param simulation - simulation instance for the sensor environment
      \return the sensor environment for the given simulation instance, nullptr if
              \p simulation hasn't been assigned one
      \sa getOrCreate
      */
      static Environment* get( agxSDK::Simulation* simulation );

      /**
      \param simulation - simulation instance for the sensor environment
      \return the sensor environment for the given simulation instance, nullptr if
              \p simulation hasn't been assigned one
      \sa getOrCreate
      */
      static const Environment* get( const agxSDK::Simulation* simulation );

    public:
      /**
      Default constructor - the default sensor systems will be added.
      */
      Environment();

      /**
      Add shape to be visible to sensors in this sensor environment. If added
      successfully, a reference to the shape will be held until the shape is
      removed or when this sensor environment is removed from its cooperative
      simulation (or explicit call to removeNotification for standalone).
      \param shape - shape instance to add
      \return true if the shape is successfully added, otherwise false
      */
      bool add( agxCollide::Shape* shape );

      /**
      Remove shape from this sensor environment. The shape will no longer be
      visible to sensors.
      \param shape - shape instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxCollide::Shape* shape );

      /**
      Add geometry to be visible to sensors in this sensor environment. All
      shapes of the given geometry will also be added and visible to sensors.
      If added successfully, a reference to the geometry will be held until
      the geometry is removed or when this sensor environment is removed from
      its cooperative simulation (or explicit call to removeNotification for standalone).
      \param geometry - geometry instance (and its shapes) to add
      \return true if the geometry is successfully added, otherwise false
      */
      bool add( agxCollide::Geometry* geometry );

      /**
      Remove geometry, and all its shapes, from this sensor environment. The
      geometry will no longer be visible to sensors.
      \param geometry - geometry instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxCollide::Geometry* geometry );

      /**
      Add rigid body to be visible to sensors in this sensor environment. All
      geometries (and shapes) of the given rigid body will also be added and
      visible to sensors. If added successfully, a reference to the rigid body
      will be held until the rigid body is removed or when this sensor environment
      is removed from its cooperative simulation (or explicit call to
      removeNotification for standalone).
      \param rb - rigid body instance (and its geometries) to add
      \return true if the rigid body is successfully added, otherwise false
      */
      bool add( agx::RigidBody* rb );

      /**
      Remove the rigid body, and all its geometries, from this sensor environment. The
      rigid body will no longer be visible to sensors.
      \param rb - rigid body instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agx::RigidBody* rb );

      /**
      Add terrain to be visible to sensors in this sensor environment. The granular
      bodies and topology changes will be reflected and visible to the sensors and
      the active granular bodies will inherit the surface material (or other) properties
      of this given terrain instance. If added successfully, a reference to the terrain
      will be held until the terrain is removed or when this sensor environment
      is removed from its cooperative simulation (or explicit call to removeNotification
      for standalone).
      \param terrain - terrain instance to add
      \return true if the terrain is successfully added, otherwise false
      */
      bool add( agxTerrain::Terrain* terrain );

      /**
      Remove terrain from this sensor environment. The terrain will no longer be visible
      to sensors.
      \param terrain - terrain instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxTerrain::Terrain* terrain);

      /**
      Add terrain pager to be visible to sensors in this sensor environment. The granular
      bodies and topology changes will be reflected and visible to the sensors and
      the active granular bodies will inherit the surface material (or other) properties
      of this given terrain pager instance. If added successfully, a reference to the terrain
      pager will be held until the terrain pager is removed or when this sensor environment
      is removed from its cooperative simulation (or explicit call to removeNotification for
      standalone).
      \param pager - terrain pager instance to add
      \return true if the terrain pager is successfully added, otherwise false
      */
      bool add( agxTerrain::TerrainPager* pager );

      /**
      Remove terrain pager from this sensor environment. The terrain pager will no longer
      be visible to sensors.
      \param pager - terrain pager instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxTerrain::TerrainPager* pager );

      /**
      Add agxCable::Cable, agxVehicle::Track, agxModel::Beam, ..., to be visible to sensors
      in this sensor environment. The wheels (agxVehicle::TrackWheel) of agxVehicle::Track
      instances are added as separate rigid bodies during this call, but can be removed
      separately if they shouldn't be visible to sensors, e.g.,
          sensorEnv->remove( wheel4->getRigidBody() ).
      If added successfully, a reference to the linked structure will be held until the linked
      structure is removed or when this sensor environment is removed from its cooperative
      simulation (or explicit call to removeNotification for standalone).
      \param linkedStructure - agxCable::Cable, agxVehicle::Track, agxModel::Beam, ..., instance to add
      \return true if the linked structure is successfully added, otherwise false
      */
      bool add( agxSDK::LinkedStructure* linkedStructure );

      /**
      Remove linked structure (agxCable::Cable, agxVehicle::Track, agxModel::Beam, ...) from this
      sensor environment. The linked structure will no longer be visible to sensors. The wheels
      of agxVehicle::Track instances will also be removed.
      \param linkedStructure - linked structure instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxSDK::LinkedStructure* linkedStructure );

      /**
      Add wire to be visible to sensors in this sensor environment. If added
      successfully, a reference to the wire will be held until the wire is
      removed or when this sensor environment is removed from its cooperative
      simulation (or explicit call to removeNotification for standalone).
      \param wire - wire instance to add
      \return true if the shape is successfully added, otherwise false
      */
      bool add( agxWire::Wire* wire );

      /**
      Remove wire from this sensor environment. The wire will no longer be visible
      to sensors.
      \param wire - wire instance to remove
      \return true if previously successfully added, otherwise false (nullptr or not previously added)
      */
      bool remove( agxWire::Wire* wire );

      /**
      Add system root node (normally an implementation of a sensor) to this
      sensor environment. If added successfully, the system will receive calls
      to 'synchronize', from the main thread, and 'execute', from the sensor
      environment thread, each step.
      \param system - system to add
      \return true if successfully added, otherwise false (nullptr or initialization failed)
      */
      bool add( SystemNode* system );

      /**
      Remove system root node (sensor) from this sensor environment.
      \param system - system to remove
      \return true if removed, otherwise false (nullptr or not present in this sensor environment)
      */
      bool remove( SystemNode* system );

      /**
      \return the raytrace scene, possible to assign ambient/atmosphere material
      */
      RtSceneRef getScene() const;

      /**
      Set environment magnetic field.
      \param magneticField - magnetic field to span the environment, nullptr removes the current
                             magnetic field
      */
      void setMagneticField( MagneticField* magneticField );

      /**
      \return the magnetic field in the sensor environment
      */
      MagneticField* getMagneticField();

      /**
      \return the magnetic field in the sensor environment
      */
      const MagneticField* getMagneticField() const;

    public:
      /**
      Destructor.
      */
      virtual ~Environment();

      DOXYGEN_START_INTERNAL_BLOCK()

      /**
      Initializes the default systems, if not previously made and spawns the environment
      thread at sleep.
      */
      virtual void addNotification() override;

      /**
      Joins the environment thread, removes all systems and added objects.
      */
      virtual void removeNotification() override;

      /**
      Steps this environment the given delta time \p dt. Blocking data synchronization
      in the current thread and integrated in the environment thread.
      \param dt - time step size
      */
      virtual void step( agx::Real dt ) override;

      /**
      Waits for the environment thread to finish its current step/integration and puts
      the environment thread to sleep. Sensor results/outputs are available after calling
      this method.
      */
      virtual void fetchStepResults() override;

      /**
      \return added objects of the given type T
      */
      template<typename T>
      agx::VectorPOD<T*> getObjects() const;

      /**
      \return first root system of the given type T
      */
      template<typename T>
      T* findRootSystem() const;

      /**
      \return all root systems of the given type T
      */
      template<typename T>
      agx::VectorPOD<T*> findRootSystems() const;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Environment );

    private:
      struct RuntimeState
      {
        enum Flag : agx::UInt32
        {
          INITIALIZED = 1 << 0,
          DO_STEP = 1 << 1,
          SYNCHRONIZING_DATA = 1 << 2,
          SLEEP = 1 << 3,
          IS_SLEEPING = 1 << 4,
          JOIN_MAIN = 1 << 30,
        };

        using Value = agx::AtomicState<Flag>;

        Value state{ (agx::UInt32)0 };
        agx::Real dt = 1.0 / 60.0;
      };

    private:
      void initialize();
      void cleanup();

      bool fireOnAdd( agx::Referenced& instance );
      bool fireOnRemove( agx::Referenced& instance );

      template<typename T>
      bool accept( bool handled, agx::Referenced& instance, bool isAdded );

      template<typename T, typename InOutHelper>
      void storeType( agxStream::OutputArchive& out,
                      InOutHelper& helper ) const;

      template<typename T, typename InOutHelper>
      void restoreType( agxStream::InputArchive& in,
                        InOutHelper& helper,
                        bool callAddOnRestored );

      bool isDone() const;
      void realize();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      SystemNodeRefVector m_systems;
      std::thread m_thread;
      RuntimeState m_runtimeData;
      agx::ReferencedHandler m_refHandler;
      RtSceneRef m_scene;
      MagneticFieldRef m_magneticField;
  };

  template<typename T>
  agx::VectorPOD<T*> Environment::getObjects() const
  {
    agx::VectorPOD<T*> objects;
    m_refHandler.template visit<T>( [&objects]( const T& instance )
    {
      objects.push_back( const_cast<T*>( &instance ) );
    } );
    return objects;
  }

  template<typename T>
  T* Environment::findRootSystem() const
  {
    for ( const auto& system : m_systems )
      if ( system->template is<T>() )
        return system->template as<T>();
    return nullptr;
  }

  template<typename T>
  agx::VectorPOD<T*> Environment::findRootSystems() const
  {
    agx::VectorPOD<T*> result;
    for ( const auto& system : m_systems )
      if ( system->template is<T>() )
        result.push_back( system->template as<T>() );
    return result;
  }

  template<typename T>
  bool Environment::accept( bool handled, agx::Referenced& instance, bool isAdded )
  {
    static_assert( std::is_base_of<agxStream::Serializable, T>::value,
                   "Unexpected that type doesn't suport serialization." );

    agxAssert( !handled || instance.template is<T>() );
    if ( handled && isAdded ) {
      agxSensor::getOrCreateEnvData( instance.template asSafe<T>() )->environment = this;
      m_refHandler.template add<T>( instance );
    }
    else if ( handled ) {
      agxSensor::getOrCreateEnvData( instance.template asSafe<T>() )->environment = nullptr;
      m_refHandler.template remove<T>( instance );
    }

    return handled;
  }

  template<typename T, typename InOutHelper>
  void Environment::storeType( agxStream::OutputArchive& out, InOutHelper& helper ) const
  {
    out << agxStream::out( "numInstances", m_refHandler.template size<T>() );
    m_refHandler.template visit<T>( [&out, &helper]( const T& instance )
    {
      out << agxStream::out( "instance", &instance );

      // By design that instances added to the ref-handler has
      // environment data.
      const EnvInternalData* envData = getEnvData( &instance );
      out << agxStream::out( "hasRtData", envData->rtData != nullptr );
      if ( envData->rtData != nullptr ) {
        envData->rtData->state.flags.store( out );
        out << agxStream::out( "rtEntityIndex", indexOf( envData->rtData->getEntityId() ) );

        helper.store( out, envData->rtData->getMaterial() );
      }
    } );
  }

  template<typename T, typename InOutHelper>
  void Environment::restoreType( agxStream::InputArchive& in,
                                 InOutHelper& helper,
                                 bool callAddOnRestored )
  {
    size_t numInstances = 0u;
    in >> agxStream::in( "numInstances", numInstances );
    for ( size_t i = 0u; i < numInstances; ++i ) {
      T* instance = nullptr;
      in >> agxStream::in( "instance", instance );
      if ( instance != nullptr ) {
        bool hasRtData = false;
        in >> agxStream::in( "hasRtData", hasRtData );
        if ( hasRtData ) {
          RtInternalData* rtData = getOrCreateRtData( instance );
          rtData->state.flags.restore( in );
          RtEntityIdIndex rtEntityIndex = 0;
          in >> agxStream::in( "rtEntityIndex", rtEntityIndex );
          rtData->setEntityId( (RtEntityId)rtEntityIndex );

          RtMaterialInstance material{};
          helper.restore( in, material );
          rtData->setMaterial( material );
        }

        if ( callAddOnRestored )
          this->add( instance );
        this->template accept<T>( true, *instance, true );
        instance->setFinished();
      }
    }
  }
}
