/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/MagneticField.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( DipoleMagneticField );

  /**
  A magnetic field as produced by a magnetic dipole.
  Default magnetic moment and dipole center are set as the relative magnetic moment and dipole
  center of Earth at the surface near the coordinates of 38.254113 degrees north and 140.875417
  degrees east.
  */
  class AGXSENSOR_EXPORT DipoleMagneticField : public MagneticField
  {
    public:
      /**
      Constructs a magnetic dipole positioned at the given \p center and using the specified
      \p magneticMoment.
      \param magneticMoment - magnetic dipole moment in world coordinates
      \param center - position of the center of the magnetic dipole in world coordinates
      */
      DipoleMagneticField( const agx::Vec3& magneticMoment =
                             agx::Vec3( 2.69e19, -7.65e22, 1.5e22 ),
                           const agx::Vec3& center =
                             agx::Vec3( -1.9e-10, 20.79e3, -6.369e6 ) );

      /**
      Set the dipole's magnetic moment.
      \param magneticMoment - magnetic dipole moment in world coordinates
      */
      void setMagneticMoment( const agx::Vec3& magneticMoment );

      /**
      \return magnetic moment of the dipole in world coordinates
      */
      const agx::Vec3& getMagneticMoment() const;

      /**
      Set the center of the dipole.
      \param center - position of the center of the magnetic dipole in world coordinates
      */
      void setCenter( const agx::Vec3& center );

      /**
      \return center position of the magnetic dipole in world coordinates
      */
      const agx::Vec3& getCenter() const;

      /**
      Calculates and returns the magnetic field at the given \p position in world coordinates.
      \param position - position in world coordinates at which the magnetic field should be
                        determined
      \return magnetic field vector at the \p position
      */
      virtual agx::Vec3 calculateMagneticField( const agx::Vec3& position ) const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::DipoleMagneticField );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::Vec3 m_magneticMoment;
      agx::Vec3 m_center;
  };
}


