/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxOpenPLX/AgxMetadata.h>

#include <openplx/Physics/Signals/InputSignal.h>

#include <memory>

namespace openplx::Physics::Signals
{
  class Output;
  class Input;
}

namespace agxopenplx
{
  class OutputSignalQueue;
  class AgxObjectMap;

  /**
   * Takes a OpenPLX input signal and updates the AGX simulation.
   * Looks up the AGX objects from a AgxObjectMap.
   */
  class InputHandler
  {
    public:
      InputHandler(std::shared_ptr<AgxObjectMap> mapper, std::shared_ptr<AgxMetadata> metadata);

      template <class InputType>
      bool handle(
        const std::shared_ptr<openplx::Physics::Signals::InputSignal>& input_signal,
        const std::shared_ptr<InputType>& input);

      template <class InputType>
      bool handleIfMatch(const std::shared_ptr<openplx::Physics::Signals::InputSignal>& input_signal)
      {
        if (input_signal == nullptr)
          return false;
        auto target = input_signal->target();
        if (target == nullptr)
          return false;
        if (auto matched_input = std::dynamic_pointer_cast<InputType>(target)) {
          return handle<InputType>(input_signal, matched_input);
        }
        return false;
      }

    private:
      template <class ModelType, class InputType, class SignalType>
      bool handleDelegated(
        const std::shared_ptr<ModelType>& model, const std::shared_ptr<InputType>& input,
        const std::shared_ptr<SignalType>& signal);

      template <class InputType, class SignalType, class... ModelType>
      inline bool delegateHandling(
        const std::shared_ptr<openplx::Physics::Signals::InputSignal>& signal, const std::shared_ptr<InputType>& input)
      {
        bool result = false;
        auto process = [&result, this](auto model, const std::shared_ptr<InputType>& input, auto signal)
        {
          if (model && signal) {
            result = this->handleDelegated(model, input, signal);
            return true;
          }
          else
            return false;
        };

        bool processedInput =
          (process(
             std::dynamic_pointer_cast<ModelType>(input->source()), input,
             std::dynamic_pointer_cast<SignalType>(signal))
           || ... || false);
        return processedInput ? result : false;
      }

    private:
      std::shared_ptr<AgxObjectMap> m_mapper;
      std::shared_ptr<AgxMetadata> m_metadata;
  };

  /**
   * Takes a OpenPLX output and queues a openplx output signal by
   * reading state from the AGX simulation.
   * Looks up the AGX objects from a AgxObjectMap.
   */
  class OutputHandler
  {
    public:
      OutputHandler(
        std::shared_ptr<OutputSignalQueue> output_queue, std::shared_ptr<AgxObjectMap> mapper,
        std::shared_ptr<AgxMetadata> metadata);

      template <class OutputType>
      bool handle(const std::shared_ptr<OutputType>& output);

      template <class OutputType>
      bool handleIfMatch(const std::shared_ptr<openplx::Physics::Signals::Output>& output)
      {
        if (auto matched_output = std::dynamic_pointer_cast<OutputType>(output)) {
          return handle<OutputType>(matched_output);
        }
        return false;
      }

    private:
      template <class ModelType, class OutputType>
      bool handleDelegated(const std::shared_ptr<ModelType>& model, const std::shared_ptr<OutputType>& output);

      template <class OutputType, class... ModelType>
      inline bool delegateHandling(const std::shared_ptr<OutputType>& output)
      {
        bool result = false;
        auto process = [&result, this](auto model, const std::shared_ptr<OutputType>& output)
        {
          if (model) {
            result = this->handleDelegated(model, output);
            return true;
          }
          else
            return false;
        };

        bool processedOutput =
          (process(std::dynamic_pointer_cast<ModelType>(output->source()), output) || ... || false);
        return processedOutput ? result : false;
      }

    private:
      std::shared_ptr<OutputSignalQueue> m_output_queue;
      std::shared_ptr<AgxObjectMap> m_mapper;
      std::shared_ptr<AgxMetadata> m_metadata;
  };
}
