/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#ifndef AGXMODEL_CONTACTMEASUREMENT_H
#define AGXMODEL_CONTACTMEASUREMENT_H

#include <map>
#include <unordered_map>

#include <agx/RigidBody.h>
#include <agxCollide/Geometry.h>
#include <agxCollide/Shape.h>
#include <agxModel/export.h>
#include <agx/Journal.h>
#include <agxData/Buffer.h>

#include <agxSDK/StepEventListener.h>
#include <agxSDK/ContactEventListener.h>

#include <agx/BitState.h>

namespace agxModel
{
  /**
  The Contact Measurement model measure impact energy on contacts between rigid bodies.
  To enable impact energy measurement add this listener to the simulation. 
    // agxSDK::Simulation *simulation;
    ContactMeasurementRef contactMeasurement = new agxModel::ContactMeasurement();
    simulation->add( contactMeasurement );

    // read impact energy from the contact measurment model

    agx::Real maxImpact = contactMeasurement->getMaxImpactEnergy(body);
  */
  
  AGX_DECLARE_POINTER_TYPES(ContactMeasurement);

  class AGXMODEL_EXPORT ContactMeasurement : public agxSDK::StepEventListener
  {
    public:
      /**
      Default constructor
      \param onlyEmittedBodies - Type of contacts to measure. Default is to measure emitted rigid body contacts
                            but the measurement could be expanded to all rigid body contacts.
      */
      ContactMeasurement(bool onlyEmittedBodies=true);

      /**
      Enable/disable debug rendering of contact points handled by this component.
      \param enable - true to enable, false to disable (Default: Disabled)
      */
      void setEnableDebugRendering( agx::Bool enable );

      /**
      \return true if debug rendering is enabled, otherwise false
      */
      agx::Bool getEnableDebugRendering() const;

      /**
      Calculate the impact energy in the post step between all rigid bodies of interreset.
      \param time - the current simulation time
      */
      virtual void pre(const agx::TimeStamp& time) override;

      /**
      Calculate the impact energy in the post step between all rigid bodies of interreset. 
      \param time - the current simulation time
      */
      virtual void post(const agx::TimeStamp& time) override;

      /**
      Called when this listener is added to the simulation.
      (given that it not already is in the simulation).
      */
      virtual void addNotification() override;

      /**
      If the simulation contains a agxModel::ContactMeasurement then that listener is returned,
      else a nullptr is returned.
      */
      static ContactMeasurement* getListener(agxSDK::Simulation* simulation);

      /**
      Return the max impact energy for a body.
      \param body - Rigid body to get max impact energy for.
      */
      agx::Real getMaxImpactEnergy(agx::RigidBody* body);

      /**
      Attach journal.
      */
      void journalAttachedCallback(agxSDK::Simulation* simulation, agx::Journal* journal);

    public:
      DOXYGEN_START_INTERNAL_BLOCK()

#ifndef SWIG
      AGXSTREAM_DECLARE_SERIALIZABLE( agxModel::ContactMeasurement );
#endif

  protected:
    void storeChangedImpacts();

    void restoreImpacts();

    void initImpactEnergyStorage();

    void createJournalBindings(agx::Journal* journal);

    agxData::Buffer* getMaxImpactBuffer();

  //////////////////////////////////////////////////////////////////////////
  // Variables
  //////////////////////////////////////////////////////////////////////////
  protected:
    agxSDK::Simulation::JournalAttachEvent::CallbackType m_journalAttachedCallback;
    agxData::BufferRef     m_maxImpactBuffer;
    agx::ComponentRef      m_dataRoot;
    agx::Uuid              m_dataRootUUID;
    agx::JournalObserver   m_journal;


    protected:
      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      void calculateImpactEnergy(const agxCollide::GeometryContactPtrVector& contacts, std::unordered_map<agx::Index, std::vector<agx::Vec3f>> preImpactSpeeds);

  
      enum StateEnum : agx::UInt32
      {
        DEBUG_RENDER = 1 << 0,
      };
      using State = agx::BitState<StateEnum, agx::UInt32>;

    private:
      agxSDK::ContactEventListenerRef m_listener;
      State m_state;   
      std::map<agx::UInt32, agx::Real> m_maxImpactEnergies;
      bool m_onlyEmittedBodies;
      bool m_clearMaxImpacts;
      int m_numNonEmittedBodies;
  };
}
#endif