/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agx/agxPhysics_export.h>
#include <agx/ref_ptr.h>
#include <agx/Referenced.h>
#include <agx/Vec3.h>
#include <agx/agx_vector_types.h>

namespace agx
{
  class Frame;
  using FrameRef = agx::ref_ptr<agx::Frame>;

  class RigidBody;
  using RigidBodyRef = agx::ref_ptr<agx::RigidBody>;
}

namespace agxControl
{

  /**
  Pure Pursuit is a geometric method for path tracking that can be used for different
  applications such as e.g. autonomous vehicles or mobile robots.

  The actor tracks a position on the path that is at a distance ahead of the current
  position and calculates the curvature (steering) that is needed to reach the tracking point.

  The choice of lookahead distance affects how well the actor will follow the path.
  A large value will lead less precise path tracking with e.g. taking shortcuts in corners.
  A small value for lookahead can lead to oscillations in the steering and many steering
  corrections, espcially when the speed increases.

  This class requires the AGX-Control license module.
  */
  class AGXPHYSICS_EXPORT PurePursuit : public agx::Referenced
  {
    public:


      /**
      Base class for Path implementations that Pure Pursuit will follow.

      There are two main methods used by pure pursuit:
      - Given a world position, find the scalar distance along the path for the closest point
      - Given a scalar value, find a position on the path

      The methods are used in the following way:
        auto s = path->findPositionOnPath(currentVehiclePos);
        auto target = path->getPointOnPath(s + lookahead);
      */
      class AGXPHYSICS_EXPORT Path : public agx::Referenced
      {
        public:
          /**
          Given a world position, find the closest point P on the path
          and return how far along the path P is.
          */
          virtual double findPositionOnPath(agx::Vec3 point) const = 0;

          /**
          Return a point on the path that is located at specified distance from the start.
          */
          virtual agx::Vec3 getPointOnPath(double distance) const = 0;
      };

      /**
      Base class for a vehicle/robot/machine for which pure pursuit should compute steering information.

      The information needed to control e.g. a car-like vehicle where front and back wheels can
      have different steering angle is different to e.g. a tracked machine where the steering
      is handled via different track speeds.
      */
      class AGXPHYSICS_EXPORT SteeringInformation : public agx::Referenced
      {
        public:
          /**
          Constructor for base representation.
          The localFrame should have the referenceBody:s model frame as parent.
          If that is not the case, the local frame will be re-parented.
          */
          explicit SteeringInformation(agx::RigidBody* referenceBody, agx::Frame* localFrame);

          /**
          Returns the RigidBody for this vehicle
          */
          agx::RigidBody* getRigidBody() const;

          /**
          Returns the reference frame for the vehicle. This is normally in model
          coordinates for the body to which the wheels or tracks are attached.
          */
          agx::Frame* getFrame() const;


          /**
          Return number of data elements that are produced by the implementation.
          */
          virtual size_t getNumOutputElements() const = 0;

          /**
          Compute steering information and write result to output buffer.

          Specializations of this class are not allowed to use a varying
          number of output elements or write more elements to outputBuffer
          than the value returned via getNumOutputElements.

          \returns true if steering information could be computed
          */
          virtual bool computeSteeringInformation(agx::Vec3 target, double* outputBuffer) = 0;

        protected:
          agx::RigidBodyRef m_body;
          agx::FrameRef m_coordinates;
      };


      /**
      Constructor for Pure Pursuit.
      \param steeringInformation Abstraction for vehicle type
      \param path Representation of the path that should be followed.
      */
      PurePursuit(PurePursuit::SteeringInformation* steeringInformation, PurePursuit::Path* path);

      virtual ~PurePursuit() = default;

      PurePursuit(const PurePursuit& other) = delete;
      PurePursuit(const PurePursuit&& other) = delete;

      PurePursuit& operator=(const PurePursuit& other) = delete;
      PurePursuit& operator=(const PurePursuit&& other) = delete;

      /**
      Returns the computed lookahead distance.
      */
      double computeLookAhead() const;

      /**
      Specify a fixed lookahead distance which must be > 0.

      This has the same effect as calling setLookAhead(anything, distance, distance)
      */
      void setLookAhead(double distance);

      /**
      Specify a velocity dependant lookahead distance.
      Resulting lookahead is vehicle base velocity multipled with velocityGain
      and then clamped to min- and max-distance.
      */
      void setLookAhead(double velocityGain, double minDistance, double maxDistance);

      /**
      Return the lookahead parameters as components in a Vec3.
      Order of components are (gain, min, max).
      */
      agx::Vec3 getLookAheadParameters() const;

      /**
      Compute steering information for the vehicle.
      Updated information can be fetched from getSteeringValues() where
      getNumOutputElements() value(s) can be read.

      For the interpretation of the values, see the
      computeSteeringInformation documentation
      matching the PurePursuit::Vechicle class being used.
      */
      bool computeSteeringInformation();

      /**
      \returns The number of computed steering values. This value
              depends on the Vehicle class in use.
      */
      size_t getNumOutputElements() const;

      /**
      \returns Pointer to a buffer where steering value(s) can be read.
      */
      const double* getSteeringValues() const;

    private:
      agx::Vec4 m_fixedBuffer;
      agx::Real m_lookaheadVelocityGain;
      agx::Real m_lookaheadMinDistance;
      agx::Real m_lookaheadMaxDistance;
      agx::ref_ptr<PurePursuit::SteeringInformation> m_vehicle;
      agx::ref_ptr<PurePursuit::Path> m_path;
      agx::RealVector m_dynamicBuffer;
  };



  /**
  Implementation of a path via waypoints and linear segments.
  */
  class AGXPHYSICS_EXPORT LinearSegmentPath : public PurePursuit::Path
  {
    public:
      explicit LinearSegmentPath(const agx::Vec3Vector& points, bool wrapAround = false);

      double findPositionOnPath(agx::Vec3 point) const override;

      agx::Vec3 getPointOnPath(double distance) const override;

    private:
      agx::Vec4Vector m_table;
      double m_totalDistance;
      bool m_wrapAround;
  };


  /**
  Implementation for a tracked vehicle.

  Computes two outputs, speed for left track and speed for right track
  given a desired center speed for the vehicle of 1 m/s.
  The computed outputs should be scaled with the same magnitude to the desired velocity.
  */
  class AGXPHYSICS_EXPORT TrackedVehicleSteeringInformation : public PurePursuit::SteeringInformation
  {
    public:
      /**
      Constructor.
      \param underCarriageBody Reference body which have a fixed alignment with the tracks.
      \param underCarriageFrame Reference frame with the following requirements:
                                - Frame should be centered between the tracks.
                                - Frame should be relative reference body model coorinates
                                - Local x-axis should be towards the right track when looking forward.
                                - Local y-axis should be forward direction for tracks
      \param trackToTrackDistance Distance between center of left and right track.
      */
      TrackedVehicleSteeringInformation(
        agx::RigidBody* underCarriageBody, agx::Frame* underCarriageFrame, double trackToTrackDistance);

      /**
      TrackedVehicleControl computes 2 values.
      */
      size_t getNumOutputElements() const override;

      /**
      This method will write 2 elements to the outputBuffer,
      speed for the left track and speed for the right track.
      */
      bool computeSteeringInformation(agx::Vec3 target, double* outputBuffer) override;

    private:
      agx::Real m_trackToTrackDistance;
  };

  /**
  Implementation for a car-like vehicle

  Computes one output, a steering angle.
  */
  class AGXPHYSICS_EXPORT CarSteeringInformation : public PurePursuit::SteeringInformation
  {
    public:
      /**
      Constructor.
      \param chassiBody Reference body which have a fixed alignment with the wheel axes.
      \param chassiFrame Reference frame with the following requirements:
                                - Frame should be centered between the wheel axes.
                                - Frame should be relative reference body model coorinates
                                - Local x-axis should be sideway direction to the right when looking forward.
                                - Local y-axis should be forward direction for car
      \param wheelbase Distance between front- and rear wheel axle.
      */
      CarSteeringInformation(agx::RigidBody* chassiBody, agx::Frame* chassiFrame, double wheelbase);

      /**
      CarControl computes 1 value.
      */
      size_t getNumOutputElements() const override;

      /**
      This method will write 1 element to the outputBuffer, the steering angle.
      */
      bool computeSteeringInformation(agx::Vec3 target, double* outputBuffer) override;

    private:
      agx::Real m_wheelBase;
  };

} // end of namespace agxControl
