#pragma once

#include <memory>
#include <string>
#include <ostream>
#include <filesystem>
#include <unordered_map>
#include <vector>
#include <utility>
#include <openplx/Token.h>
#include <openplx/NodeVisitor.h>
#include <openplx/Types.h>

namespace openplx
{
    class Document;

    /**
     * Takes a syntax tree with or without line and column information
     * and overwrites the line and column information of all tokens
     * with sensible values corresponding to a "good"-ish layout.
     *
     * This will never be as powerful as for example black (a python code formatter)
     * but we do our best.
     *
     * After layout the syntax tree can be converted to a string with the openplx::Formatter
     * class.
     */
    class LayoutEngine : private NodeVisitor {
        public:
            explicit LayoutEngine(bool layout_expressions = false, int64_t indent = 4);
            void layoutDocument(const std::shared_ptr<Document>& document);

            void addPathReplacement(std::filesystem::path from, std::string to);

        private:

            bool moveComment(line_t previous_line, line_t new_line);

            void replacePathInConstant(const ConstantPtr& constant);

            void visitDocument(const DocPtr& document) override;

            void visitModelDeclaration(const ModelDeclPtr& model_declaration) override;

            void visitVarAssignment(const VarAssignPtr& var_assignment) override;

            void visitMethodDeclaration(const MethodDeclPtr& method_declaration) override;

            void visitAnnotation(const AnnotationPtr& annotation) override;

            void visitImport(const ImportPtr& ptr) override;

            void visitTraitImpl(const TraitImplPtr& trait_impl) override;
            bool m_layout_expressions;
            col_t m_indent;
            line_t m_line;
            line_t m_previous_line;
            col_t m_column;
            std::unordered_map<line_t, Token> m_comments;
            std::vector<std::pair<std::filesystem::path, std::string>> m_path_replacements;
    };
}
