/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <set>
#include <memory>

#include <agx/HashTable.h>
#include <agx/Vec3.h>
#include <agx/Vector.h>
#include <agx/agx_vector_types.h>


DOXYGEN_START_INTERNAL_BLOCK()
class SQEM;
typedef std::shared_ptr<SQEM> SQEMRef;
DOXYGEN_END_INTERNAL_BLOCK()

namespace agxUtil {
    
    /**
    A skeleton with spherical joints. Instances of this class are returned by the SphereSkeletoniser class and the "skeletoniseMesh" function.
    */
    class SphereSkeleton {
    public:
        /**
        A joint in a sphereskeleton.
        */
        struct Joint {
            agx::Size index = 0;
            //The index of the vertex from which this joint originates from. Used when identifying nodes in the SphereSkeletoniser class.
            agx::Size skeletoniserIndex = 0;
            agx::Vec3 position;
            agx::Real radius = 0;
            agx::VectorPOD<agx::Size> adjJoints;
        };
        agx::Vector<Joint> joints;
        agx::Size leafJoint = 0;        

        /**
        Segment the skeleton into a set of new skeletons, splitting the skeleton at each junction and separating disjoint skeletons.
        */
        AGXPHYSICS_EXPORT agx::Vector<SphereSkeleton> segmentSkeleton() const;       

        /**
        Creates a new skeleton from the longest continous path in the given skeleton.
        \return A new skeleton only containing the lnogst continous path in this skeleton.
        */
        AGXPHYSICS_EXPORT SphereSkeleton getLongestContinuousSkeletonSegment();

        /**
		Copy constructor
		\param other The skeleton to copy
        */
        AGXPHYSICS_EXPORT SphereSkeleton(const SphereSkeleton& other);

		/**
		Default constructor
		*/
        AGXPHYSICS_EXPORT SphereSkeleton() { }

        class dfs_iterator {        
        friend class SphereSkeleton;
        public:
            /**
            Creates an invalid dfs_iterator. Use SphereSkeleton::begin to construct a new iterator.
            */
            AGXPHYSICS_EXPORT dfs_iterator();

            /**
            Copy constructor
			\param other The iterator to copy
            */
            AGXPHYSICS_EXPORT dfs_iterator(const dfs_iterator& other);

            /**
            Increment the iterator to the next joint
            */
            AGXPHYSICS_EXPORT dfs_iterator& inc();


            AGXPHYSICS_EXPORT dfs_iterator& operator++();

            AGXPHYSICS_EXPORT dfs_iterator next() const;

            /**
            \returns A pointer to the joint who's traversal discovered the current joint. Null if this == skeleton.begin()
            */
            AGXPHYSICS_EXPORT Joint* prev_joint() const;

            /**
            \returns A pointer to the next joint in the traversal order. Null if this.isLast()
            */
            AGXPHYSICS_EXPORT Joint* next_joint() const;

            AGXPHYSICS_EXPORT dfs_iterator operator++(int);

            AGXPHYSICS_EXPORT Joint& deref() const;

            AGXPHYSICS_EXPORT Joint& operator*() const;

            AGXPHYSICS_EXPORT Joint* operator->() const;

            AGXPHYSICS_EXPORT agx::Bool equals(const dfs_iterator& rhs) const;

            AGXPHYSICS_EXPORT bool operator==(const dfs_iterator& rhs) const;

            AGXPHYSICS_EXPORT bool operator!=(const dfs_iterator& rhs) const;

            AGXPHYSICS_EXPORT agx::Bool isEnd() const;

			/**
            \returns True if the current joint is the last joint to be traversed
            */
            AGXPHYSICS_EXPORT agx::Bool isLast() const;

            /**
            \returns True if the current joint only has one adjacent joint
            */
            AGXPHYSICS_EXPORT agx::Bool isLeaf() const;

			/**
			\returns True if the current joint has more than two adjacent joints
			*/
            AGXPHYSICS_EXPORT agx::Bool isBranch() const;

            /**
			\returns True if the current joint is equal to skeleton.begin()
            */
            AGXPHYSICS_EXPORT agx::Bool isBegin() const;
        private:
            dfs_iterator(const SphereSkeleton* skeleton, agx::Size idx);

            void traverse(agx::Size idx);

            const SphereSkeleton* skeleton;
            agx::Size currJoint;
            agx::Size prevJoint;
            agx::HashSet<agx::Size> marked;
            agx::Vector<std::pair<agx::Size, agx::Size>> toTraverse;
        };

		/**
		\returns A depth first search iterator starting at the leaf joint in the skeleton
		*/
        AGXPHYSICS_EXPORT dfs_iterator begin() const;

        /**
		\returns A depth first search iterator in the end state. This iterator is invalid and should not be dereferenced.
        */
        AGXPHYSICS_EXPORT dfs_iterator end() const;

		/**
		\param skeletoniserIndex The skeletoniser index to test
		\returns True if the given skeletoniser index is present in the skeleton.
		*/
        AGXPHYSICS_EXPORT bool contains(agx::UInt32 skeletoniserIndex);
};

    /**
    Utilises the SphereSkeletoniser class to generate a SphereSkeleton from a given trimesh structure.
    \param vertices The vertices of the mesh
    \param indices The indices of the mesh (always 3 per triangle, see agxCollide/Trimesh.h)
    */
    AGXPHYSICS_EXPORT SphereSkeleton skeletoniseMesh(const agx::Vec3Vector& vertices, const agx::UInt32Vector& indices);
        
    /**
    A class capable of producing a sphere-skeleton from a trimesh through more granular operations. 
    */
    class SphereSkeletoniser {
    public:        
        /**
        Initialise a skeletoniser which will work on a given mesh.
        \param vertices The vertices of the mesh
        \param indices The indices of the mesh (always 3 per triangle, see agxCollide/Trimesh.h)
        */
        AGXPHYSICS_EXPORT SphereSkeletoniser(const agx::Vec3Vector& vertices, const agx::UInt32Vector& indices);
        /**
        Sets the measure of much the skeletoniser prioritises collapsing faces. Higher values means a more aggressive skeletonisation but can lead
        to faulty joints in complex models. Lower prioritises simplifying the original shape of the mesh which can lead to loss of quality in the resulting skeleton.
        \param devalueFactor The factor to devalue collapses which do not lead to a strictly skeletal structure with. Default = 10.
        */
        AGXPHYSICS_EXPORT void setFaceDevalueFactor(agx::Real devalueFactor);
        /**
        Collapses the mesh until a skeletal structure is reached.
        */
        AGXPHYSICS_EXPORT void collapseUntilSkeleton();
        /**
        Collapses the next optimal choice of edge
        \return True if an edge could be collapsed
        */
        AGXPHYSICS_EXPORT bool collapseNext();
        /**
        \return The remaining vertices on the possibly non-manifold "meso-skeleton" contained in the skeletoniser
        */
        AGXPHYSICS_EXPORT agx::Size remainingVertices();
        /**
        \return The remaining edges on the possibly non-manifold "meso-skeleton" contained in the skeletoniser
        */
        AGXPHYSICS_EXPORT agx::Size remainingEdges();
        /**
        \return The remaining triangles/faces on the possibly non-manifold "meso-skeleton" contained in the skeletoniser
        */
        AGXPHYSICS_EXPORT agx::Size remainingTriangles();
        /**
        \return The current resolution (joints per length) of the meso-skeleton contained in the skeletoniser
        */
        AGXPHYSICS_EXPORT double currentSkeletonResolution();

        /**
        Applies the average radius of the remaining vertices to all the vertices.
        */
        AGXPHYSICS_EXPORT void applyAverageRadius();

        /**
        Applies the \p radius as the radius of the remaining vertices.
        \param radius The radius to apply
        */
        AGXPHYSICS_EXPORT void applyRadius(agx::Real radius);

        enum UpscalingMethod {
            JOINT,
            EDGE, 
            BOTH
        };
        /**
        Performs a pass over all edges in the current structure and attempts to upscale each edge.
        \param targetNumJoints The target number of nodes in the skeleton.
        \param method The means of upscaling. Can be per joint, per edge, or a combination of both.
        \param skeletonFilter Optional parameter for filtering which vertices are considered. Only edges and vertices present in \p skeletonFilter will be upscaled.
        \pre The meso-skeleton contained is a full skeleton with no faces left.
        */
        AGXPHYSICS_EXPORT void upscaleSkeleton(agx::Size targetNumJoints, UpscalingMethod method = UpscalingMethod::BOTH, SphereSkeleton* skeletonFilter = nullptr);

        /**
        Performs a pass over the nodes in the skeleton in the skeletoniser's current structure and attempts to collapse any nodes which sit below the given threshold in distance.
        \param skeleton The skeleton who's joints will be consolidated in the skeletoniser
        \param radiusFactor Factor by which to multiply the average radius between two nodes considered to produce the distance threshhold.
        */
        AGXPHYSICS_EXPORT void consolidateSkeleton(SphereSkeleton& skeleton, double radiusFactor = 2);

        /**
        Checks if upscaling is possible on the given edge.
        \param v0 Start of edge to check
        \param v1 End of edge to check
        \returns True or false
        */
        AGXPHYSICS_EXPORT bool isUpscalePossible(agx::UInt32 v0, agx::UInt32 v1);

        /**
        Checks if upscaling is possible on the given vertex
        \param splitVertex The vertex at which to check
        \returns True or false
        */
        AGXPHYSICS_EXPORT bool isUpscalePossible(agx::UInt32 splitVertex);

        /**
        Create a new vertex in the meso-skeleton from the combined regions of an edge.
        \param v0 The id of the first vertex in the edge
        \param v1 The id of the second vertex in the edge
        \pre The meso-skeleton contained is a full skeleton with no faces left.
        */
        AGXPHYSICS_EXPORT bool upscaleEdge(agx::UInt32 v0, agx::UInt32 v1, SphereSkeleton* skeletonFilter = nullptr);

        /**
        \param splitVertex The id of the vertex which will have its region split into two vertices
        \pre The meso-skeleton contained is a full skeleton with no faces left.
        */
        AGXPHYSICS_EXPORT bool upscaleJoint(agx::UInt32 splitVertex, SphereSkeleton* skeletonFilter = nullptr);        

        /**
        Removes the joint with the given index from the skeletoniser. Removes data and does not merge faces or regions. Collapsing or upscaling after a removal
        will negatively affect the results of the skeletonisation in regards to the mesh.
        \param indexToRemove The index in the joint vector at which to remove a joint.        
        \param cut True if the edges from the joint should be deleted. True if the edges should be assigned to the closest neighbour.
        */
        AGXPHYSICS_EXPORT void removeVertex(const agx::Size indexToRemove, const bool cut = false);        

        /**
        Constructs a sphere-skeleton from the current "meso-skeleton". Will ignore any faces/triangles present in the meso-skeleton
        to guarantee a skeletal structure.        
        \return A SphereSkeleton object containing the joints of the skeleton with adjacency data
        */
        AGXPHYSICS_EXPORT SphereSkeleton getSkeleton();

        /**
        Returns a new vector containing all of the original vertices associated with the skeletoniser vertex with id \p id
        \param id The id of the skeletoniser vertex
        \returns A vector of points.
        */
        AGXPHYSICS_EXPORT agx::Vec3Vector getSurfacePoints(agx::UInt32 id);

        /**
        Shorthand for upscaling a by a single joint
		\param method The means of upscaling. Can be per joint, per edge, or a combination of both.
        */
        AGXPHYSICS_EXPORT void addJoint(UpscalingMethod method = UpscalingMethod::BOTH);
    private:
        struct Edge {
            agx::UInt32 v0;
            agx::UInt32 v1;

            Edge();
            Edge(agx::UInt32 v0, agx::UInt32 v1);

            bool operator <(const Edge& other) const;

            bool operator >(const Edge& other) const;

            bool operator ==(const Edge& other) const;

            bool operator !=(const Edge& other) const;

            agx::UInt32 hash() const;

            bool contains(agx::UInt32 id);
        };

        struct EdgeData {
            SQEMRef sqem;
            agx::Vec3 sphereCenter;
            agx::Real sphereRadius = 0;
            agx::Real cost = 0;
        };

        struct EdgeWithCost {
            Edge edge;
            agx::Real cost;

            EdgeWithCost(Edge edge, agx::Real cost);
            EdgeWithCost(agx::UInt32 v0, agx::UInt32 v1, agx::Real cost);

            bool operator ==(const Edge& e);

            bool operator ==(const EdgeWithCost& e);

            bool operator <(const EdgeWithCost& b) const;

            bool operator >(const EdgeWithCost& b) const;
        };

        struct Triangle {
            agx::UInt32 id;
            agx::Vec3u32 indices;
            bool deleted;

            Triangle();
            Triangle(const Triangle& other);

            Triangle(agx::UInt32 id, agx::UInt32 v0, agx::UInt32 v1, agx::UInt32 v2);
            Triangle(agx::UInt32 id, agx::Vec3u32 vec);

            void set(size_t i, agx::UInt32 value);

            void replace(agx::UInt32 target, agx::UInt32 value);

            bool contains(agx::UInt32 vId);

            bool isDegenerate();

            agx::UInt32 operator[](size_t i) const;
            agx::UInt32 hash() const;
        };

        struct SurfacePoint {
            agx::Vec3 position;
            agx::Vec3 normal;
        };

        struct BarycentricCellPoint {
            SurfacePoint point;
            agx::Real sourceTriangleArea;
            agx::UInt32 sourceTriangleIndex;
        };

        struct RegionVertex {
            agx::Size indexOnMesh;
            SurfacePoint originalMeshPoint;
            agx::VectorPOD<BarycentricCellPoint> barycentricCell;
        };
        
        static constexpr int NUM_PRESAMPLED_DIRECTIONS = 32;
        struct Vertex {
            agx::UInt32 id;
            agx::Vec3 position;
            agx::Real radius;
            agx::Vector<RegionVertex> regionVertices;
            SQEMRef sqem;

            agx::HashSet<agx::UInt32> adjacentFaces;
            agx::HashSet<agx::UInt32> adjacentVertices;

            //Minimum and maximum extents in each direction for the region this vertex represents
            std::pair<agx::Real, agx::Real> directionalWidthIntervals[NUM_PRESAMPLED_DIRECTIONS];

            bool deleted;

            Vertex(agx::UInt32 id, agx::Size indexOnMesh, const SQEM& initialSqem, agx::VectorPOD<BarycentricCellPoint>& barycentricCell, agx::Vec3 initialPosition, agx::Vec3 initialNormal, agx::HashSet<agx::UInt32>& adjacentFaces, agx::HashSet<agx::UInt32>& adjacentVertices);
            Vertex(agx::UInt32 id, agx::Size numRegionVerts, RegionVertex regionVertices[], agx::HashSet<agx::UInt32>& adjacentVertices, agx::Real& minIntervalWidth, agx::Real& maxIntervalWidth);
            Vertex();
            bool operator == (const Vertex& v);
            agx::Real evaluate();
            agx::Real evaluate(agx::Vec3 pos, agx::Real radius);
        };
        
        void updateEdgeData(Edge edge, EdgeData& data);
        Edge popCheapestEdge();
        void collapseEdge(Edge edge);
        double currentSkeletonLength();
        double avgVertexRadius();
        double meanSquaredDistanceToEdges(agx::VectorPOD<agx::Vec3*>& points, agx::HashSet<Edge>& edges);
        void sampleRegionVertexPoints(agx::VectorPOD<agx::Vec3*>& samples, agx::VectorPOD<RegionVertex*> regionVertices);
        int getFurthestNeighbours(agx::UInt32 dst[2], agx::UInt32 vertexId, std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        agx::Real calculateEdgeFitScore(Edge edge);
        agx::Vec2 addNewVertexFromRegion(agx::Vec3 estimatedPosition, double radius, agx::HashSet<agx::UInt32>& adjacentVertices, agx::Size numRegionVertices, RegionVertex* regionVertexArray);
        agx::Vec2 reConstructVertex(Vertex& v, agx::Size numRegionVertices, RegionVertex* regionVertexArray);
        void reArrangeRegionVertices(Edge edge, std::function<bool(agx::UInt32)> rejectionCriteria);
        void reArrangeRegionVertices(std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        void sortRegionVerticesAlongSkeleton(agx::Vector<RegionVertex>& dst, Edge e, std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        bool upscaleVertex(agx::UInt32 splitVertex, std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        bool upscaleEdge(Edge edge, std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        Vertex& findNeighbour(Vertex& v, std::function<bool(agx::UInt32)> rejectionCriteria = nullptr);
        bool shouldUseCoarseRadiusApproximation();
        void mergeEdges(Vertex& dst, Vertex& src);
        void mergeRegions(Vertex& dst, Vertex& src);

        agx::Vector<Vertex> m_vertices;
        agx::HashSet<agx::UInt32> m_remainingVertexIds;        
        agx::Vector<Triangle> m_triangles;
        agx::Size m_numTriangles;
        agx::HashTable<Edge, EdgeData> m_edgeData;
        std::set<EdgeWithCost> m_edgePriorityQueue;
        agx::Real m_faceDevalueFactor;
        
        static agx::Vec3 s_preSampledUnitSphereDirections[NUM_PRESAMPLED_DIRECTIONS];        
    };
}