/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxTerrain/export.h>
#include <agxRender/Color.h>
#include <agx/Vec3.h>
#include <agx/agx_vector_types.h>
#include <numeric>

#include <agx/Physics/ParticleGeometryContactEntity.h>


DOXYGEN_START_INTERNAL_BLOCK()
namespace detail
{
  struct ContactPatchData;
  struct ContactSpanData;
}

namespace agxTerrain
{
  class TerrainToolCollection;
  class Terrain;
  class Shovel;

  /**
  Struct that contains information about how to partition the shovel into
  different contact patches.
  */
  struct AGXTERRAIN_EXPORT ContactPartitionData
  {
    agx::RealVector angles;
    agx::Vec3Vector normals;
    agx::Vec3Vector points;

    agx::UInt getNumPatches() const;

    void clear() {
      angles.clear();
      normals.clear();
      points.clear();
    }
  };

  /**
  Struct that contains information about the span of all the particle-geometry
  contacts between the shovel and the terrain particles.
  */
  struct ContactSpanData
  {
  public:
    /**
    The angle span of the contact points projected in shovel cross-section plane between the
    vector from the plane midpoint to cutting edge and the vector from the midpoint to the highest contact.
    */
    agx::RealVector angleSpan = { -1.0, -1.0 };

    /**
    The distance span of the contact patch along the cutting edge.
    */
    agx::RealVector distanceSpan = { 0.0, 0.0 };

    /**
    The points of the distance span along the cutting edge.
    */
    agx::Vec3Vector widthPointSpan = { { 0, 0, 0 }, { 0, 0, 0 } };

    agx::Physics::ParticleGeometryContactPtr topPoint1;

    agx::Physics::ParticleGeometryContactPtr topPoint2;

    /**
    Check if the distance span data is valid for a certain index.
    */
    bool isValid( size_t index ) const
    {
      /**
      If both distance span and point span is zero, data was not written to this index.
      This means that we cannot construct a plane from this data that we use to project shovel
      contact points.
      */
      return !agx::equalsZero( distanceSpan[ index ] )
             && !agx::equivalent( widthPointSpan[ index ], agx::Vec3( 0, 0, 0 ) );
    }
  };


  /**
  Struct that contains information about the contact patch restructuring of the shovel-aggregate contacts.
  Each contact patch consists of 2 contact points ( derived from some of the contact points in the patch area )
  and a normal ( which is the mean normal of all particle-geometry normals in the contact patch region ).
  */
  class ContactPatch
  {
  public:
    ContactPatch()
    {
    }

    agx::Vec3 getAveragePoint() const {
      return std::accumulate( m_points.begin(),
                              m_points.end(),
                              agx::Vec3() ) / static_cast< agx::Real >( m_points.size() );
    }

    agx::Vec3f getNormal()
    {
      return std::accumulate( m_normals.begin(),
                              m_normals.end(),
                              agx::Vec3f()).normal();
    }

    agx::UInt getNumPoints() const { return m_points.size(); }

    void setDefaultPoint1( const agx::Physics::ParticleGeometryContactPtr& contactPoint1 ) { m_defaultPoint1 = contactPoint1; }

    void setDefaultPoint2( const agx::Physics::ParticleGeometryContactPtr& contactPoint2 ) { m_defaultPoint2 = contactPoint2; }

    agx::Physics::ParticleGeometryContactPtr getDefaultPoint1() const { return m_defaultPoint1; }

    agx::Physics::ParticleGeometryContactPtr getDefaultPoint2() const { return m_defaultPoint2; }

    void addNormal( const agx::Vec3f& normal ) { m_normals.push_back( normal ); }

    void addPoint( const agx::Vec3& point ) { m_points.push_back( point ); }

  private:
    agx::Vec3Vector  m_points;
    agx::Vec3fVector m_normals;

    agx::Physics::ParticleGeometryContactPtr  m_defaultPoint1;
    agx::Physics::ParticleGeometryContactPtr  m_defaultPoint2;
  };

  typedef agx::Vector<ContactPatch> ContactPatchVector;

  /**
  Create contact patch partitioning data given the specified shovel and it's settings.
  */
  AGXTERRAIN_EXPORT ContactPartitionData createContactPartitions( agxTerrain::Shovel* shovel );

  /**
  \return a predetermined patch debug color given a contact patch index.
  */
  AGXTERRAIN_EXPORT agxRender::Color getPatchDebugColor( size_t index );
}

DOXYGEN_END_INTERNAL_BLOCK()
