/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/SystemNode.h>
#include <agxSensor/RaytraceHandles.h>
#include <agxSensor/RaytraceShapeInstance.h>

#include <agxWire/Wire.h>

#include <agxSDK/LinkedStructure.h>

#include <agxTerrain/Terrain.h>
#include <agxTerrain/TerrainPager.h>

#include <memory>
#include <vector>

namespace agxSensor
{
  struct RtVec3fVector;

  /**
  Raytrace system representing agxCollide::Shape instances as mesh(es)
  in the raytrace environment. The transforms of the shapes are synchronized
  each step.
  */
  class AGXSENSOR_EXPORT RtShapeCollisionShapeSystem : public SystemNode
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      virtual ~RtShapeCollisionShapeSystem() = default;

      virtual bool onAdd( agx::Referenced& instance ) override;
      virtual bool onRemove( agx::Referenced& instance ) override;
      virtual void synchronize( const SystemNode::CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtShapeCollisionShapeSystem );

    private:
      struct RtShapeInstanceData
      {
        bool isValid() const
        {
          return shape != nullptr;
        }

        RtShapeInstance instance = { nullptr, nullptr, nullptr };
        const agxCollide::Shape* shape = nullptr;
      };

    private:
      using RtShapeInstanceDataVector = std::vector<RtShapeInstanceData>;

    private:
      bool contains( const agxCollide::Shape* shape ) const;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      RtShapeInstanceDataVector m_rtShapeInstances;
  };

  /**
  Raytrace system representing agxSDK::LinkedStructure (agxCable::Cable, agxVehicle::Track
  agxModel::Beam, ...) instances as shapes in the raytrace environment. The transforms are
  synchronized each step.
  */
  class AGXSENSOR_EXPORT RtShapeLinkedStructureSystem : public SystemNode
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      virtual ~RtShapeLinkedStructureSystem() = default;

      virtual bool onAdd( agx::Referenced& instance ) override;
      virtual bool onRemove( agx::Referenced& instance ) override;
      virtual void synchronize( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtShapeLinkedStructureSystem );

    private:
      struct LsData
      {
        struct RtShapeInstanceScale
        {
          RtShapeInstance instance;
          agx::Vec3 scale;
        };

        using RtShapeInstanceScaleVector = std::vector<RtShapeInstanceScale>;

        const agxSDK::LinkedStructure* linkedStructure = nullptr;
        RtEntityId entityId{ NullRtEntityId };
        RtMaterialInstance material{};
        RtShapeInstanceScaleVector shapeData; /**< Matches the total number of shapes rather than number of segments. */
      };

      using LsDataVector = std::vector<LsData>;
      using ShapeScaleRtShapeTable = agx::HashTable<agx::Vec3, RtShapeRef>;
      using ShapeTypeShapeScaleRtShapeCache = std::array<ShapeScaleRtShapeTable, agxCollide::Shape::NUM_TYPES>;

    private:
      bool contains( const agxSDK::LinkedStructure* linkedStructure ) const;
      void initialize( agxCollide::Shape& shape,
                       LsData::RtShapeInstanceScale& shapeScale,
                       RtInstanceData* shapeInstanceData );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      LsDataVector m_lsData;
      ShapeTypeShapeScaleRtShapeCache m_shapeCache;
  };

  /**
  Raytrace system representing agxWire::Wire instances as meshes in the raytrace environment.
  */
  class AGXSENSOR_EXPORT RtShapeWireSystem : public SystemNode
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      RtShapeWireSystem();

    public:
      virtual ~RtShapeWireSystem() = default;

      virtual bool onAdd( agx::Referenced& instance ) override;
      virtual bool onRemove( agx::Referenced& instance ) override;
      virtual void synchronize( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtShapeWireSystem );

    private:
      struct WireSegmentData
      {
        RtShapeInstance halfSphere;
        RtShapeInstance cylinder;
      };
      using WireSegmentDataBuffer = std::vector<WireSegmentData>;

      struct WireData
      {
        const agxWire::Wire* wire = nullptr;
        WireSegmentDataBuffer segmentDataBuffer;
        RtMaterialInstance material{};
        RtEntityId entityId{ NullRtEntityId };
        size_t numSegments = 0;
      };

      using WireDataVector = std::vector<WireData>;

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      bool contains( const agxWire::Wire* wire ) const;

    private:
      WireDataVector m_wireData;
      RtShapeRef m_halfSphereMesh;
      RtShapeRef m_cylinderMesh;
  };

  /**
  Base raytrace system handling ordinary and paged agxTerrain::Terrain instances.
  */
  class AGXSENSOR_EXPORT RtShapeTerrainSystemBase : public SystemNode
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      virtual ~RtShapeTerrainSystemBase() = default;

    protected:
      struct ParticleTerrainId
      {
        RtMaterialInstance material{};
        RtEntityId entityId{ NullRtEntityId };
      };
      using SurfaceMaterialTable = agx::HashTable<const agx::Material*, ParticleTerrainId>;

      struct RtTerrainData
      {
        RtTerrainData();
        bool isValid() const
        {
          return terrain != nullptr;
        }

        RtShapeInstance instance = { nullptr, nullptr, nullptr };
        const agxTerrain::Terrain* terrain = nullptr;
        std::shared_ptr<RtVec3fVector> vertices;
      };

      static RtTerrainData create(RtSceneRef scene, agxTerrain::Terrain* terrain, RtInstanceData* rtData = nullptr);
      static void updateHeights(RtTerrainData& terrainData);

    protected:
      void updateParticles( const agx::ParticleSystem* particleSystem,
                            const SurfaceMaterialTable& materialTable );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      using ParticleShapeInstances = std::vector<RtShapeInstance>;

    private:
      RtShapeRef m_sphereShape;
      ParticleShapeInstances m_particleInstances;
  };

  /**
  Raytrace system handling agxTerrain::Terrain instances.
  */
  class AGXSENSOR_EXPORT RtShapeTerrainSystem : public RtShapeTerrainSystemBase
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      virtual bool onAdd(agx::Referenced& instance) override;
      virtual bool onRemove(agx::Referenced& instance) override;
      virtual void synchronize( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtShapeTerrainSystem );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      bool contains(const agxTerrain::Terrain* terrain) const;

    private:
      using RtTerrainDataVector = std::vector<RtTerrainData>;
      RtTerrainDataVector m_rtTerrainInstances;
  };

  /**
  Raytrace system handling paged agxTerrain::Terrain instances.
  */
  class AGXSENSOR_EXPORT RtShapeTerrainPagerSystem : public RtShapeTerrainSystemBase
  {
    public:
      DOXYGEN_START_INTERNAL_BLOCK()

      virtual bool onAdd(agx::Referenced& instance) override;
      virtual bool onRemove(agx::Referenced& instance) override;
      virtual void synchronize( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::RtShapeTerrainPagerSystem );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      struct RtTerrainPagerData
      {
        bool isValid() const
        {
          return terrainPager != nullptr;
        }

        using RtTerrainDataMap = agx::HashVector<const agxTerrain::Terrain*, RtTerrainData>;
        const agxTerrain::TerrainPager* terrainPager = nullptr;
        RtTerrainDataMap terrainData;
      };

      bool contains(const agxTerrain::TerrainPager* pager) const;
      bool syncronizeTiles(RtTerrainPagerData& data, SurfaceMaterialTable& materialTable);

    private:
      using RtTerrainPagerDataVector = std::vector<RtTerrainPagerData>;
      RtTerrainPagerDataVector m_rtTerrainPagerData;
  };
}
