/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxSensor/RaytraceSurfaceMaterial.h>

namespace agxSensor
{
  /**
  Opaque surface material reflecting the light according to an explicitly specified bidirectional 
  reflectance distribution function (BRDF).

  This surface reflection model can correctly model all opaque homogeneous surfaces, given an 
  explicitly specified BRDF of sufficient resolution.

  The BRDF samples provided to this material should be arranged as in four dimensional array 
  flattened in memory where the indices represent the following:
  
  - [0]: Sample index in theta (vertical) direction for the light incidence direction
  - [1]: Sample index in phi (horizontal) direction for the light incidence direction
  - [2]: Sample index in theta (vertical) direction for the view direction
  - [3]: Sample index in phi (horizontal) direction for the view direction

  Additionally, the samples should represent a non-isotropic BRDF, thus specifying the full phi 
  range from 0 to 2*pi. Theta ranges from 0 to pi/2.

  A complete 2 x 2 x 2 x 2 explicit BRDF could thus be structured as,
  ```
  std::vector<float>{
    0.50f, 0.25f, 1.00f, 0.50f,
    0.25f, 0.00f, 0.50f, 0.25f,
    1.00f, 0.50f, 0.25f, 0.13f,
    0.50f, 0.25f, 0.13f, 0.00f
  };
  ```
  where each sub-quadrant thus represents view direction configuration and the sub-quadrants 
  together represent light incidence configuration.
  */
  class AGXSENSOR_EXPORT RtBrdfExplicitMaterial : public RtSurfaceMaterial
  {
  public:
    static constexpr RtMaterialHandle::Type Type = RtMaterialHandle::Type::OPAQUE_BRDF_EXPLICIT;

  public:
    static RtBrdfExplicitMaterial create();

    /**
    \return the surface material associated with the given shape
    */
    static RtBrdfExplicitMaterial get( const agxCollide::Shape* shape );

    /**
    \return the surface material associated with the given geometry
    */
    static RtBrdfExplicitMaterial get( const agxCollide::Geometry* geometry );

    /**
    \return the surface material associated with the given rigid body
    */
    static RtBrdfExplicitMaterial get( const agx::RigidBody* rb );

    /**
    \return the surface material associated with the given linked structure (agxCable::Cable, agxModel::Beam, agxVehicle::Track, ...)
    */
    static RtBrdfExplicitMaterial get( const agxSDK::LinkedStructure* linkedStructure );

    /**
    \return the surface material associated with the given wire
    */
    static RtBrdfExplicitMaterial get( const agxWire::Wire* wire );

    /**
    \return the surface material associated with the given terrain
    */
    static RtBrdfExplicitMaterial get( const agxTerrain::Terrain* terrain );

    /**
    \return the surface material associated with the given terrain pager
    */
    static RtBrdfExplicitMaterial get( const agxTerrain::TerrainPager* terrainPager );

  public:
    RtBrdfExplicitMaterial( RtMaterialInstance instance );

    /**
    \return the amount of samples in the light theta (vertical) direction
    */
    size_t getResolutionThetaLight() const;

    /**
    \return the amount of samples in the light phi (horizontal) direction
    */
    size_t getResolutionPhiLight() const;

    /**
    \return the amount of samples in the view theta (vertical) direction
    */
    size_t getResolutionThetaView() const;

    /**
    \return the amount of samples in the view phi (horizontal) direction
    */
    size_t getResolutionPhiView() const;

    /**
    \return the BRDF samples
    */
    std::vector<float> getSamples() const;

    /**
    Attempt to reshape the sample array to the specified dimensions. The size of the underlying 
    memory is not changed and thus the specified dimensions must resolve to the same total memory
    footprint.
    \param resolutionThetaLight - resolution in light theta (vertical) direction
    \param resolutionPhiLight - resolution in light phi (horizontal) direction
    \param resolutionThetaView - resolution in view theta (vertical) direction
    \param resolutionPhiView - resolution in view phi (horizontal) direction
    \return true if the reshape is a success or false if the specified reshape dimensions do not 
            produce a total memory footprint size equal to that of the previous configuration
    */
    bool reshape( size_t resolutionThetaLight, size_t resolutionPhiLight, 
                  size_t resolutionThetaView, size_t resolutionPhiView );

    /**
    Set the explicit bidirectional reflection distribution function (BRDF) samples.
    \param resolutionThetaLight - resolution in light theta (vertical) direction
    \param resolutionPhiLight - resolution in light phi (horizontal) direction
    \param resolutionThetaView - resolution in view theta (vertical) direction
    \param resolutionPhiView - resolution in view phi (horizontal) direction
    \param samples - BRDF samples of size (resolutionThetaLight x resolutionPhiLight x 
           resolutionThetaView x resolutionPhiView)
    */
    RtBrdfExplicitMaterial setSamples( size_t resolutionThetaLight, size_t resolutionPhiLight,
                                       size_t resolutionThetaView, size_t resolutionPhiView, 
                                       const std::vector<float>& samples );

    /**
    Attempt to read the BRDF samples from the specified luminance image file.
    The image values flattened in memory represent the, flattened in memory-, four dimensional array 
    of BRDF samples.
    \param filename - path to the image file containing the BRDF samples
    \param resolutionThetaLight - resolution in light theta (vertical) direction
    \param resolutionPhiLight - resolution in light phi (horizontal) direction
    \param resolutionThetaView - resolution in view theta (vertical) direction
    \param resolutionPhiView - resolution in view phi (horizontal) direction
    \return true if the BRDF sample import is successful or false if specified dimensions are 
            incorrect or if the image file cannot be read
    */
    bool setSamplesFromImageFile( const agx::String& filename, size_t resolutionThetaLight,
                                  size_t resolutionPhiLight, size_t resolutionThetaView, 
                                  size_t resolutionPhiView );

  public:
    void store( agxStream::OutputArchive& out ) const;
    void restore( agxStream::InputArchive& in );

  private:
    using RtSurfaceMaterial::RtSurfaceMaterial;
  };
}
