/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once
#include <agxSensor/RaytraceMaterial.h>

namespace agxSensor
{
  /**
  Ambient homogeneous atmospheric material attenuating the light and producing stochastic returns in 
  the atmosphere.

  This atmospheric material model is suitable for most common weather conditions, atmospheric 
  aerosols, atmospheric powders and other cases where the atmosphere can be considered homogeneous 
  due to the small scale of the disturbance particles.

  This model is not suitable for significantly inhomogeneous atmospheres, such as soot from fires, 
  exhaust from vehicles or smokestacks, or tropospheric clouds.
  */
  class AGXSENSOR_EXPORT RtAmbientMaterial : public RtMaterial
  {
    public:
      static constexpr RtMaterialHandle::Type Type = RtMaterialHandle::Type::AMBIENT_ATMOSPHERE;

    public:
      static RtAmbientMaterial create();

    public:
      RtAmbientMaterial( RtMaterialInstance instance );

      /**
      \return the material refractive index
      */
      float getRefractiveIndex() const;

      /**
      Assign new refractive index to the material.
      \param refractiveIndex - refractive index
      */
      RtAmbientMaterial setRefractiveIndex( float refractiveIndex );

      /**
      \return the attenuation coefficient of the material
      */
      float getAttenuationCoefficient() const;

      /**
      Assign new attenuation coefficient to the material.
      \param alpha - attenuation coefficient
      */
      RtAmbientMaterial setAttenuationCoefficient( float alpha );

      /**
      \return the scaling of the atmospheric return probability
      */
      float getReturnProbabilityScaling() const;

      /**
      Assign new scaling of the atmospheric return probability. This value indicates the probability 
      scaling of the atmospheric return gamma distribution defined by the shape and scale 
      parameters.
      \param A - scaling of the atmospheric return probability
      */
      RtAmbientMaterial setReturnProbabilityScaling( float A );

      /**
      \return shape parameter (k) of the atmospheric return gamma distribution
      */
      float getReturnGammaDistributionShapeParameter() const;

      /**
      Assign new shape parameter of the atmospheric return gamma distribution.
      \param k - gamma distribution shape parameter
      */
      RtAmbientMaterial setReturnGammaDistributionShapeParameter( float k );

      /**
      \return scale parameter (theta) of the atmospheric return gamma distribution
      */
      float getReturnGammaDistributionScaleParameter() const;

      /**
      Assign new scale parameter of the atmospheric return gamma distribution.
      \param theta - gamma distribution shape parameter
      */
      RtAmbientMaterial setReturnGammaDistributionScaleParameter( float theta );

      /**
      Configure the material parameters as clear weather air with the specified visibility. This
      configuration uses fog-like atmospheric returns combined with (inverse) MOR attenuation.
      \param visibility - visibility in kilometers
      */
      RtAmbientMaterial configureAsAir( float visibility );

      /**
      Configure the material parameters as foggy weather with the specified visibility. This
      configuration uses fog-like atmospheric returns combined with Al-Naboulsi attenuation.
      \param visibility - visibility in kilometers
      \param wavelength - sensor wavelength in nanometers
      \param maritimeness - interpolation value between continental (0.0) and maritime (1.0) fog
      */
      RtAmbientMaterial configureAsFog( float visibility, float wavelength, 
                                        float maritimeness = 0.0f );

      /**
      Configure the material parameters as rainfall with the specified precipitation rate. This
      configuration uses rain-like atmospheric returns combined with Carbonneau rainfall 
      attenuation.
      \param rate - precipitation rate in mm/h
      \param tropicalness - interpolation value between light (0.0) and tropical (1.0) rain
      */
      RtAmbientMaterial configureAsRainfall( float rate, float tropicalness = 0.0f );

      /**
      Configure the material parameters as light snowfall with the specified precipitation rate. 
      This configuration uses rain-like atmospheric return combined with Carbonneau snowfall 
      attenuation.
      \param rate - precipitation rate in mm/h
      \param wavelength - sensor wavelength in nanometers
      */
      RtAmbientMaterial configureAsSnowfall( float rate, float wavelength );

    public:
      void store( agxStream::OutputArchive& out ) const;
      void restore( agxStream::InputArchive& in );

    private:
      using RtMaterial::RtMaterial;
  };
}
