/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxSensor/export.h>

#include <agx/Referenced.h>

#include <agxStream/Serializable.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( LidarProperties );

  /**
  Properties/parameters of a lidar model.
  */
  class AGXSENSOR_EXPORT LidarProperties : public agx::Referenced, public agxStream::Serializable
  {
    public:
      /**
      Construct a properties configuration for a lidar.
      \param beamDivergence - divergence angle of the beam in radians
      \param beamExitRadius - radius of the beam at emission in meters
      */
      LidarProperties( agx::Real beamDivergence = 1.0E-3,
                       agx::Real beamExitRadius = 5.0E-3 );

      /**
      Divergence of the lidar laser light beam, in radians.
      This the total "cone angle", i.e. twice the half-apex angle.
      The beam divergence affects the intensity calculation when using the
      INTENSITY_F32 field.
      */
      agx::Real getBeamDivergence() const;

      /**
      Set beam divergence. See also getBeamDivergence().
      */
      void setBeamDivergence( agx::Real divergence );

      /**
      The radius of the lidar laser light beam as it exits the lidar, in meters.
      The beam exit radius affects the intensity calculation when using the
      INTENSITY_F32 field.
      */
      agx::Real getBeamExitRadius() const;

      /**
      Set exit radius. See also setBeamExitRadius().
      */
      void setBeamExitRadius( agx::Real exitRadius );

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::LidarProperties );

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      struct Beam
      {
        agx::Real divergence;
        agx::Real exitRadius;
      };

    private:
      Beam m_beam;
  };
}
