/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/IMUOutput.h>
#include <agxSensor/SensorOutputHandler.h>

#include <optional>

namespace agxSensor
{
  using OptionalUniqueId = std::optional<size_t>;
  using OptionalIMUOutputMappnig = std::optional<std::pair<size_t, IMUOutput*>>;

  AGX_DECLARE_POINTER_TYPES( IMUOutputHandler );

  /**
  Handler responsible for how the IMU readings are assembled and provided to the user.
  */
  class AGXSENSOR_EXPORT IMUOutputHandler : public SystemNode, public ISensorOutputHandler
  {
    public:
      /**
      Default constructor.
      */
      IMUOutputHandler();

      /**
      Add the given IMU output to the output handler mapped to the given unique id.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \param output - IMU output instance to add
      \return true if \p uniqueId > 0, \p uniqueId is unique and \p output != nullptr
      */
      bool add( size_t uniqueId, IMUOutput* output );

      /**
      Add the given IMU output to the output handler. The added IMUOutput can later be accessed
      from get() by using the returned uniqueId.
      \param output - IMU output instance to add
      \return the unique id assigned to the output on success, or nullopt on failure
      */
      OptionalUniqueId add( IMUOutput* output );

      /**
      Build matching IMU output to a given data type T and add it mapped to the given unique id.
      This method verifies the size of the sum of data fields matches the size of T.
      \param uniqueId - unique id > 0, id 0 is reserved and invalid to use
      \return true if \p uniqueId > 0 and \p uniqueId is unique
      */
      template<typename T, IMUOutput::Field... field>
      bool add( size_t uniqueId );

      /**
      Build matching IMU output to a given data type T. This method verifies the size of the sum of
      data fields matches the size of T. The created IMUOutput returned from this method can later
      be accessed from get() by using the returned uniqueId.
      \return a pair containing the unique id assigned to the output and the built IMU output
      */
      template<typename T, IMUOutput::Field... field>
      OptionalIMUOutputMappnig add();

      /**
      Access IMU output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      IMUOutput* get( size_t uniqueId );

      /**
      Access IMU output instance given the unique id from when it was added.
      \param uniqueId - unique id for the output instance
      \return output instance for the given unique id if it exists, otherwise nullptr
      */
      ISensorOutput* getBase( size_t uniqueId ) override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::IMUOutputHandler );

      DOXYGEN_END_INTERNAL_BLOCK()

    protected:
      using OutputTable = agx::HashTable<size_t, IMUOutputRef>;

    private:
      OutputTable m_outputs;
  };

  template<typename T, IMUOutput::Field... field>
  bool IMUOutputHandler::add( size_t uniqueId )
  {
    IMUOutputRef output = new IMUOutput();
    output->build<field...>();
    return add( uniqueId, output );
  }

  template<typename T, IMUOutput::Field... field>
  OptionalIMUOutputMappnig IMUOutputHandler::add()
  {
    IMUOutputRef output = new IMUOutput();
    output->build<field...>();

    auto uniqueId = add( output );
    if ( uniqueId )
      return std::make_pair( *uniqueId, output );
    else
      return std::nullopt;
  }
}
