/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once
#include <agxSensor/Environment.h>
#include <agxSensor/FrameAttachedSensor.h>
#include <agxSensor/GyroscopeModel.h>
#include <agxSensor/GyroscopeOutputHandler.h>

#include <agx/Frame.h>

#include <agxSDK/Simulation.h>

namespace agxSensor
{
  AGX_DECLARE_POINTER_TYPES( Gyroscope );
  AGX_DECLARE_VECTOR_TYPES( Gyroscope );

  /**
  Gyroscope instance class defined by a frame/transform and a model. Any output from the sensor
  is accessed through the GyroscopeOutputHandler.
  */
  class AGXSENSOR_EXPORT Gyroscope : public FrameAttachedSensor
  {
    public:
      /**
      Find gyroscope given name.
      \param simulation - simulation with a sensor environment
      \param name - name of gyroscope instance to search for
      \return first gyroscope instance with name \p name if found, otherwise nullptr
      */
      static Gyroscope* find( const agxSDK::Simulation* simulation, const agx::Name& name );

      /**
      Find gyroscope given name.
      \param environment - sensor environment
      \param name - name of gyroscope instance to search for
      \return first gyroscope instance with name \p name if found, otherwise nullptr
      */
      static Gyroscope* find( const Environment* environment, const agx::Name& name );

      /**
      Find all gyroscopes, given name.
      \param simulation - simulation with a sensor environment
      \param name - name of gyroscope instance to search for
      \return gyroscope instances with name \p name
      */
      static GyroscopePtrVector findAll( const agxSDK::Simulation* simulation,
                                         const agx::Name& name );

      /**
      Find all gyroscopes, given name.
      \param environment - sensor environment
      \param name - name of gyroscope instance to search for
      \return gyroscope instances with name \p name
      */
      static GyroscopePtrVector findAll( const Environment* environment, const agx::Name& name );

      /**
      Find all gyroscopes.
      \param simulation - simulation with a sensor environment
      \return all gyroscope instances in the sensor environment
      */
      static GyroscopePtrVector findAll( const agxSDK::Simulation* simulation );

      /**
      Find all gyroscopes.
      \param environment - sensor environment
      \return all gyroscope instances in the given sensor environment \p environment
      */
      static GyroscopePtrVector findAll( const Environment* environment );

    public:
      /**
      Construct this gyroscope attached to the given \p frame and configured by the \p model.
      \note This gyroscope instance is invalid if the given \p model is nullptr.
      \param frame - parent frame to attach this gyroscope to, may be dynamic, the output is
                     relative to this frame. If nullptr, a new frame is created aligned to the world
                     frame
      \param model - model defining the parameters of this gyroscope instance, invalid if nullptr
      */
      Gyroscope( agx::Frame* frame, GyroscopeModel* model );

      /**
      Assign a new parent frame to attach this gyroscope to. The effects will take place the next
      time the environment is stepped.
      \param frame - new frame for this gyroscope
      */
      virtual void setFrame( agx::Frame* frame ) override;

      /**
      \return the frame of this gyroscope, defining this gyroscope -> world transform
      */
      virtual agx::Frame* getFrame() const override;

      /**
      \return the model of this gyroscope
      */
      GyroscopeModel* getModel() const;

      /**
      \return the output handler managing the outputs from the gyroscope
      */
      GyroscopeOutputHandler* getOutputHandler() const;

      /**
      \return the output handler managing the outputs from the gyroscope
      */
      virtual ISensorOutputHandler* getOutputHandlerBase() const override;

      DOXYGEN_START_INTERNAL_BLOCK()

    public:
      virtual void result( const CallbackData& data ) override;
      virtual void cleanup() override;

      AGXSTREAM_DECLARE_SERIALIZABLE( agxSensor::Gyroscope );

    protected:
      Gyroscope();

      DOXYGEN_END_INTERNAL_BLOCK()

    private:
      agx::FrameRef m_frame;
      GyroscopeModelRef m_model;
  };

}
