/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/
#pragma once

#include <agx/agxPhysics_export.h>
#include <agx/Referenced.h>
#include <agx/HashVector.h>
#include <agx/AffineMatrix4x4.h>
#include <agxCollide/Space.h>

#include <agxRender/RenderProxy.h>
#include <agxRender/Renderable.h>
#include <agxRender/Graph.h>

namespace agxCollide
{
  class Sphere;
  class Box;
  class Plane;
  class Convex;
  class Plane;
  class Cylinder;
  class Cone;
  class Line;
  class HollowCylinder;
  class HollowCone;
}

namespace agxRender
{

  AGX_DECLARE_POINTER_TYPES(RenderProxyFactory);
  /**
  Abstract class which is responsible for returning a pointer to a specified RenderProxy type.
  This class is to be implemented by the user, depending on the rendering system.
  The implementation of this factory might have some own cache of Proxies, or not.

  This class is also responsible for creating a class derived from agxRender::Graph::GraphRenderer.
  */
  class AGXPHYSICS_EXPORT RenderProxyFactory : public agx::Referenced
  {
  public:
    RenderProxyFactory();

    /// Interface for creating and returning a SphereProxy
    virtual SphereProxy* createSphere(float radius) = 0;

    /// Interface for creating and returning a SphereProxy
    virtual SphereProxy* createSphere(agxCollide::Sphere* shape) = 0;

    /// Interface for creating and returning a BoxProxy
    virtual BoxProxy* createBox(const agx::Vec3& halfExtents) = 0;

    /// Interface for creating and returning a BoxProxy
    virtual BoxProxy* createBox(agxCollide::Box* shape) = 0;

    /// Interface for creating and returning a LineProxy
    virtual LineProxy* createLine(const agx::Vec3& p1, const agx::Vec3& p2) = 0;

    /// Interface for creating and returning a LineProxy
    virtual LineProxy* createLine(agxCollide::Line* shape) = 0;

    /// Interface for creating and returning a CylinderProxy
    virtual CylinderProxy* createCylinder(float radius, float height) = 0;

    /// Interface for creating and returning a CylinderProxy
    virtual CylinderProxy* createCylinder(agxCollide::Cylinder *shape) = 0;

    /// Interface for creating and returning a HollowCylinderProxy
    virtual HollowCylinderProxy* createHollowCylinder(float outerRadius, float height, float thickness) = 0;

    /// Interface for creating and returning a HollowCylinderProxy
    virtual HollowCylinderProxy* createHollowCylinder(agxCollide::HollowCylinder *shape) = 0;

    /// Interface for creating and returning a ConeProxy
    virtual ConeProxy* createCone(float radius, float height) = 0;

    /// Interface for creating and returning a ConeProxy
    virtual TruncatedConeProxy* createCone(agxCollide::Cone* shape) = 0;

    /// Interface for creating and returning a TruncatedConeProxy
    virtual TruncatedConeProxy* createTruncatedCone(float topRadius, float bottomRadius, float height) = 0;

    /// Interface for creating and returning a HollowConeProxy
    virtual HollowConeProxy* createHollowCone(float topRadius, float bottomRadius, float height, float thickness) = 0;

    /// Interface for creating and returning a HollowCone
    virtual HollowConeProxy* createHollowCone(agxCollide::HollowCone *shape) = 0;

    /// Interface for creating and returning a CapsuleProxy
    virtual CapsuleProxy* createCapsule(float radius, float height) = 0;

    /// Interface for creating and returning a CapsuleProxy
    virtual CapsuleProxy* createCapsule(agxCollide::Capsule* shape) = 0;

    /// Interface for creating and returning a WireShapeProxy
    virtual agxRender::WireShapeProxy* createWireShape(float radius, float height,
      const agx::Vec3& previousEndPoint0, const agx::Vec3& previousEndPoint1) = 0;

    /// Interface for creating and returning TextProxy
    virtual TextProxy* createText(const agx::String& text, const agx::Vec3& pos, float size = 1) = 0;

    /// Interface for creating and returning PlaneProxy
    virtual PlaneProxy* createPlane(const agx::Vec3& normal, agx::Real distance) = 0;

    /// Interface for creating and returning PlaneProxy
    virtual PlaneProxy* createPlane(agxCollide::Plane* shape) = 0;

    /// Interface for creating and returning HeightfieldProxy
    virtual HeightFieldProxy* createHeightfield(agxCollide::HeightField* shape) = 0;

    /// Interface for creating and returning ContactsProxy
    virtual agxRender::ContactsProxy* createContacts(const agxCollide::GeometryContactPtrVector&) = 0;

    /// Interface for creating and returning batch rendering of rigid bodies (cm).
    virtual agxRender::RigidBodyBatchRenderProxy* createRigidBodies(const agx::RigidBodyPtrSetVector* /*enabledBodies*/) = 0;

    virtual agxRender::WireRenderProxy* createWire(float radius, const agx::Vec3& color) = 0;

    /**
    Set a transformation which could be applied to all transformations for RenderProxy's.
    This can be used to transform all objects to the center of the earth instead of trying to render on the surface of the earth (6000km).
    This transform is by default I.
    If you for example do m.setTranslate(-10,0,0) all RenderProxy's will be rendered translated -10 units in X.
    If a RenderManager is present, RenderManager::update() will be called
    */
    virtual void setGlobalTransform(const agx::AffineMatrix4x4&) {}


    virtual RenderProxy* createSphereBatchRenderer(
      agxData::Buffer* positions,
      agxData::Buffer* rotations,
      agxData::Buffer* radii,
      agxData::Buffer* colors,
      agxData::Buffer* enableRendering,
      agxData::Value* bound,
      agx::Component* context) = 0;

    /// Interface for creating and returning TrimeshProxy
    virtual agxRender::TrimeshProxy* createTrimesh(agxCollide::Trimesh* mesh) = 0;

    /// Create a SphereProxy without arguments
    SphereProxy* create(float radius);

    /// Create a BoxProxy without arguments
    BoxProxy* create(const agx::Vec3& halfExtents);

    /// Create a LineProxy without arguments
    LineProxy* create(const LineProxy*);

    /// Create a CylinderProxy without arguments
    CylinderProxy* create(const CylinderProxy*);

    /// Create a CylinderProxy without arguments
    HollowCylinderProxy* create(const HollowCylinderProxy*);

    /// Create a ConeProxy without arguments
    ConeProxy* create(const ConeProxy*);

    /// Create a TruncatedConeProxy without arguments
    TruncatedConeProxy* create(const TruncatedConeProxy*);

    /// Create a HollowConeProxy without arguments
    HollowConeProxy* create(const HollowConeProxy*);

    /// Create a CapsuleProxy without arguments
    CapsuleProxy* create(const CapsuleProxy*);

    /// Create a TextProxy without arguments
    TextProxy* create(const TextProxy*);

    /// Create a PlaneProxy without arguments
    PlaneProxy* create(const PlaneProxy*);

    /// Create a ContactsProxy without arguments
    agxRender::ContactsProxy* create(const agxCollide::GeometryContactPtrVector& contacts);

    /// Create a rigid body batch render proxy
    agxRender::RigidBodyBatchRenderProxy* create(const agx::RigidBodyPtrSetVector* enabledBodies);

    /**
    This method should create and return an implementation of a GraphRenderer, a class that can render the statistics graphs
    as described in the class Graph::GraphRenderer.
     \return a pointer to the GraphRenderer
    */
    virtual Graph::GraphRenderer* getGraphRenderer() = 0;

    /**
    Set the default RenderMode that will be used when a new RenderProxy is created.
    \param mode - New RenderMode
    */
    void setDefaultRenderMode(RenderProxy::RenderMode mode);

    /**
    \return The RenderMode used when new RenderProxy's are created
    */
    RenderProxy::RenderMode getDefaultRenderMode() const;

    void setEnable(bool flag);
    bool getEnable() const;

  protected:
    friend class RenderManager;

    virtual void setRenderManager(RenderManager* mgr) {
      m_mgr = mgr;
    }

    virtual ~RenderProxyFactory();
    RenderManager* m_mgr;
    RenderProxy::RenderMode m_defaultRenderMode;
    bool m_enabled;
  };

  inline void RenderProxyFactory::setEnable(bool flag)
  {
    m_enabled = flag;
  }
  inline bool RenderProxyFactory::getEnable() const
  {
    return m_enabled;
  }
}
