/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <openplx/Error.h>
#include <openplx/Physics/Geometries/Material.h>
#include <openplx/Physics/Interactions/SurfaceContact/Model.h>
#include <openplx/Physics/Interactions/Interaction.h>
#include <openplx/Physics/Interactions/Dissipation/DefaultFriction.h>
#include <openplx/Physics3D/Interactions/MateConnector.h>
#include <openplx/Physics3D/Geometries/TriMesh.h>
#include <openplx/Physics3D/Bodies/Body.h>
#include <openplx/Physics3D/Bodies/RigidBody.h>
#include <openplx/Physics3D/Bodies/Inertia.h>
#include <openplx/Terrain/TerrainMaterial.h>
#include <agxOpenPLX/export.h>

namespace openplx::agxerror
{
  constexpr openplx::error_code_t ModelWasNotSimulatable = 1;
  constexpr openplx::error_code_t MissingConnectedBody = 2;
  constexpr openplx::error_code_t AgxIsNotInitialized = 3;
  constexpr openplx::error_code_t InvalidMateConnectorAxis = 4;
  constexpr openplx::error_code_t PathNotAbsolute = 5;
  constexpr openplx::error_code_t InvalidObjFile = 6;
  constexpr openplx::error_code_t InvalidFrictionModel = 7;
  constexpr openplx::error_code_t DuplicateAnnotation = 8;
  constexpr openplx::error_code_t InvalidFrictionFrame = 9;
  constexpr openplx::error_code_t InvalidTriMesh = 10;
  constexpr openplx::error_code_t MissingGeometryMassTensor = 11;
  constexpr openplx::error_code_t NegativeMass = 12;
  constexpr openplx::error_code_t InvalidInertiaTensor = 13;
  constexpr openplx::error_code_t DuplicateMaterialPairForSurfaceContactModelDefinition = 14;
  constexpr openplx::error_code_t MisplacedMateConnector = 15;
  constexpr openplx::error_code_t MissingTrait = 16;
  constexpr openplx::error_code_t MisplacedBody = 17;
  constexpr openplx::error_code_t MissingTerrainMaterialConfiguration = 18;
  constexpr openplx::error_code_t MaterialsMustBeUnique = 19;
}

#ifdef _MSC_VER
  #pragma warning(push)
  #pragma warning( disable : 4251)  // warning C4251: class X needs to have dll-interface to be used by clients of class Y
#endif

namespace agxopenplx {
#define AGX_OPENPLX_BASIC_ERROR(name, code)                                                                                                         \
  class AGXOPENPLX_EXPORT name : public openplx::Error                                                                                                                \
  {                                                                                                                                                 \
    public:                                                                                                                                         \
      static constexpr openplx::error_code_t ErrorCode = code;                                                                                      \
      name(openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column, const std::string& source_id); \
                                                                                                                                                    \
    protected:                                                                                                                                      \
      virtual std::string createErrorMessage() const override;                                                                                      \
  };

  AGX_OPENPLX_BASIC_ERROR(ModelWasNotSimulatable, openplx::agxerror::ModelWasNotSimulatable)
  AGX_OPENPLX_BASIC_ERROR(AgxIsNotInitialized, openplx::agxerror::AgxIsNotInitialized)
  AGX_OPENPLX_BASIC_ERROR(DuplicateAnnotation, openplx::agxerror::DuplicateAnnotation)
  AGX_OPENPLX_BASIC_ERROR(MissingTrait, openplx::agxerror::MissingTrait)

  #undef AGX_OPENPLX_BASIC_ERROR

  class AGXOPENPLX_EXPORT InvalidMateConnectorAxis : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidMateConnectorAxis;
      InvalidMateConnectorAxis(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Interactions::MateConnector> mate_connector);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Interactions::MateConnector> m_mate_connector;
  };

  class AGXOPENPLX_EXPORT MissingConnectedBody : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MissingConnectedBody;
      MissingConnectedBody(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics::Interactions::Interaction> interaction);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics::Interactions::Interaction> m_interaction;
  };

  class AGXOPENPLX_EXPORT PathNotAbsolute : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::PathNotAbsolute;
      PathNotAbsolute(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column, const std::string& source_id,
        const std::string& path);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::string m_path;
  };

  class AGXOPENPLX_EXPORT MisplacedMateConnector : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MisplacedMateConnector;
      MisplacedMateConnector(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Interactions::MateConnector> mate_connector);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Interactions::MateConnector> m_mate_connector;
  };

  class AGXOPENPLX_EXPORT MisplacedBody : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MisplacedBody;
      MisplacedBody(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Bodies::Body> body);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Bodies::Body> m_body;
  };

  class AGXOPENPLX_EXPORT MissingTerrainMaterialConfiguration : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MissingTerrainMaterialConfiguration;
      MissingTerrainMaterialConfiguration(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Terrain::TerrainMaterial> material);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Terrain::TerrainMaterial> m_material;
  };

  class AGXOPENPLX_EXPORT MaterialsMustBeUnique : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MaterialsMustBeUnique;
      MaterialsMustBeUnique(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics::Geometries::Material> material);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics::Geometries::Material> m_material;
  };

  class AGXOPENPLX_EXPORT InvalidObjFile : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidObjFile;
      InvalidObjFile(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, const std::string& file);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::string m_file;
  };


  class AGXOPENPLX_EXPORT InvalidTriMesh : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidTriMesh;
      InvalidTriMesh(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Geometries::TriMesh> trimesh);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Geometries::TriMesh> m_trimesh;
  };

  class AGXOPENPLX_EXPORT InvalidFrictionModel : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidFrictionModel;
      InvalidFrictionModel(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id,
        std::shared_ptr<openplx::Physics::Interactions::SurfaceContact::Model> surface_model);

    protected:
      virtual std::string createErrorMessage() const override;

      std::shared_ptr<openplx::Physics::Interactions::SurfaceContact::Model> m_surface_model;
  };

  class AGXOPENPLX_EXPORT InvalidFrictionFrame : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidFrictionFrame;
      InvalidFrictionFrame(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id,
        std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultFriction> friction_model);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics::Interactions::Dissipation::DefaultFriction> m_friction_model;
  };

  class AGXOPENPLX_EXPORT MissingGeometryMassTensor : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::MissingGeometryMassTensor;
      MissingGeometryMassTensor(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Bodies::RigidBody> rigid_body);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Bodies::RigidBody> m_rigid_body;
  };

  class AGXOPENPLX_EXPORT NegativeMass : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::NegativeMass;
      NegativeMass(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Bodies::Inertia> inertia);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Bodies::Inertia> m_inertia;
  };

  class AGXOPENPLX_EXPORT InvalidInertiaTensor : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode = openplx::agxerror::InvalidInertiaTensor;
      InvalidInertiaTensor(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id, std::shared_ptr<openplx::Physics3D::Bodies::Inertia> inertia);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics3D::Bodies::Inertia> m_inertia;
  };

  class AGXOPENPLX_EXPORT DuplicateMaterialPairForSurfaceContactModelDefinition : public openplx::Error
  {
    public:
      static constexpr openplx::error_code_t ErrorCode =
        openplx::agxerror::DuplicateMaterialPairForSurfaceContactModelDefinition;
      DuplicateMaterialPairForSurfaceContactModelDefinition(
        openplx::line_t from_line, openplx::col_t from_column, openplx::line_t to_line, openplx::col_t to_column,
        const std::string& source_id,
        std::shared_ptr<openplx::Physics::Interactions::SurfaceContact::Model> surface_model);

    protected:
      virtual std::string createErrorMessage() const override;

    private:
      std::shared_ptr<openplx::Physics::Interactions::SurfaceContact::Model> m_surface_model;
  };
}

#ifdef _MSC_VER
  #pragma warning(pop)
#endif
