/*
Copyright 2007-2025. Algoryx Simulation AB.

All AGX source code, intellectual property, documentation, sample code,
tutorials, scene files and technical white papers, are copyrighted, proprietary
and confidential material of Algoryx Simulation AB. You may not download, read,
store, distribute, publish, copy or otherwise disseminate, use or expose this
material unless having a written signed agreement with Algoryx Simulation AB, or having been
advised so by Algoryx Simulation AB for a time limited evaluation, or having purchased a
valid commercial license from Algoryx Simulation AB.

Algoryx Simulation AB disclaims all responsibilities for loss or damage caused
from using this software, unless otherwise stated in written agreements with
Algoryx Simulation AB.
*/

#pragma once

#include <agxPowerLine/PowerLine.h>
#include <agxPowerLine/RotationalDimension.h>

namespace agxDriveTrain
{

  /**
  Single disk plate clutch with dry friction.  Input and output shafts are
  connected to plates with rough surfaces.  A spring loaded mechanism increases
  the normal force between the plates between a minimum and maximum, giving rise
  to the concept of torque capacity.  When the plates are at maximum distance
  from each other, no force is transmitted.  When they are at zero distance,
  maximum pressure is applied.

  Together with Coulomb friction, this keeps the plates rotating at nearly the
  same angular velocity untill the stick-slip transition is reached,
  alternately, when the torque required to move the output shaft at the same
  velocity as the input shaft reaches the torque capacity.  This happens when
  the load on the output shaft is too difficult to move.  If the two disks move
  relative to each other, this is called slipping so one can read the slip
  velocity at any time.

  The "opening fraction" here represents the value of the torque capacity in
  proportion to the maximum capacity, i.e., when the plates are fully closed,
  i.e., maximum normal force is achieved.  As for most clutches, this model
  includes a "locked" mode in which a clamp is activated to prevent any slip.
  This stays "ON" until the clutch is disengaged, or if the output shaft starts
  to move faster but in the same direction as the input shaft, as in the case of
  dynamic breaking.

  This model has two modes, namely, manual and automatic.  In the manual mode,
  the user is responsible to adjust the opening fraction, or set the lock
  mechanism directly. This can be done using a control algorithm and an event
  listener.  For the automatic mode, a time constant determines how long it
  takes to go from fully open to fully engaged.  The function "engage"
  determines whether we are ramping up or down.  Setting auto lock mode
  activates logic to activate or deactivate the lock according to simple logic.
  If the slip velocity is small in proportion of the input shaft velocity, the
  lock is activated.  If either "engage(false)" function is called or the torque
  between the places is negative -- dynamic braking -- then the lock is
  deactivated.


  The maximum available torque before slipping for a given opening fraction
  \f$f\f$ is `\f$fT_c\f$' where \f$T_c\f$ is the torque capacity defined by

  \f[
    T_c = 2\frac{2}{3}\mu\frac{r_2^3-r_1^3}{r_2^2-r_1^2}F
  \f]

  where \f$r_i\f$ are the outer and inner radii.  This defaults to a `reasonable'
  number corresponding to a passenger car, namely, around 40 Nm

  Automatic engagement makes the fraction increase linearly from the time the
  "engage(true)" function is called, or decreases linearly from the time the
  "engage(false)" is called.

  We assume that the pressure plate is a linear spring so that $F$ is
  directly proportional to it's position.

  */
  class AGXMODEL_EXPORT DryClutch : public agxPowerLine::RotationalConnector
  {
  public:
    DryClutch();

    /**
     Puts the clutch pedal in assisted mode so it will fully engage or disengage in a
     time period determined by setTimeConstant below. Using engage(true) will turn
     manual mode to false as per the rule: respect the users's intention.
    \param engage - true to engage, false to disengage from current fraction (Default: false).
    */
    void setEngage(bool engage);

    /**
    \return - Returns whether the clutch is engaged or not.
    */
    bool isEngaged() const;

    /**
    \param fraction - Set the fraction of the clutch pedal opening in range [0,1] (Default: 0.0).
    */
    void setFraction(agx::Real fraction);

    /**
    \return - Returns the fraction of the clutch pedal opening (fraction of maximum normal
    force betwen the friction plates).
    */
    agx::Real getFraction() const;

    /**
    Set the time constant for the clutch engagement. Engagement
    is linear and proceeds by a fraction of +/- step/tau per step. The default
    time constant is in seconds.
    \param tau - the time constant. The unit of tau is the same as the unit of time step size defined by the user (Default: 2.5).
    */
    void setEngageTimeConstant(agx::Real tau);

    /**
    Set the time constant for clutch disengagement. Disengagement
    is linear and proceeds by a fraction of +/- step/tau per step.   The default
    time constant is in seconds.
    \param tau - the time constant. The unit of tau is the same as the unit of time step size defined by the user (Default: 1.0).
    */
    void setDisengageTimeConstant(agx::Real tau);

    /**
    When manual mode is true, user must control all aspects of the clutch
    engagement and disengagement.  Using the engage method will turn this flag to false.
    \param manual - Set or unset manual mode (Default: false).
    */
    void setManualMode(bool manual);

    /**
    \return - Returns true in manual mode, false otherwise.
    */
    bool isManualMode() const;

    /**
    \return - Returns the time constant of the clutch pedal for engaging.
    */
    agx::Real getEngageTimeConstant() const;

    /**
    \return - Returns the time constant of the clutch pedal for disengaging.
    */
    agx::Real getDisengageTimeConstant() const;

    /**
    Lock or unlock the friction plates. In locked mode, the two plates are fixed rigidly together.
    \param setLock - desired state (Default: false).
    */
    void setLock(bool setLock);

    /**
    Returns the lock status.
    */
    bool isLocked() const;

    /**
    This is the maximum torque that the friction plates can transfer: it will slip if
     this MinRelativeSlip is exceeded. Set the torque capacity directly here, or specify the
     geometrical and physical parameters of the friction plates, allowing the program to 
     automatically calculate the torque capacity (Default: 2000). 
     */
    void setTorqueCapacity(agx::Real capacity);

    /**
    \return - Returns the torque capacity of the friction plates.
    */
    agx::Real getTorqueCapacity() const;

    /**
    In auto lock mode, the friction plates will lock when the slip is small enough in
    proportion to the velocity of the input shaft.
    \param autoLock - true or false to enable or disable autoLock, respectively (Default: false).
    */
    void setAutoLock(bool autoLock);

    /**
    \return - State of auto lock.
    */
    bool isAutoLock() const;

    /**
    \return - Torque transferred between the friction plates.
    */
    agx::Real getTorque() const;

    /**
    \return - Relative velocity inbetween the friction plates.
    */
    agx::Real getSlip() const;

    /**
    \return - Angular velocity of input shaft.
    */
    agx::Real getInputVelocity() const;

    /**
    \return - Angular velocity of output shaft.
    */
    agx::Real getOutputVelocity() const;

    /**
    In autolock mode, the friction plates will lock with the ratio of slip to input
    shaft velocity is less than this threshold.
    \param minRelativeSlip - the threshold of the minimum slip (Default: 1e-5).
    */
    void  setMinRelativeSlip(agx::Real minRelativeSlip);

    /**
    \return - Minimum relative velocity for lock.
    */
    agx::Real getMinRelativeSlip() const;

    DOXYGEN_START_INTERNAL_BLOCK()


    AGXSTREAM_DECLARE_SERIALIZABLE(agxDriveTrain::DryClutch);

    /**
    Stores internal data into stream.
    */
    virtual bool store(agxStream::StorageStream& str) const override;

    /**
    Restores internal data from stream.
    */
    virtual bool restore(agxStream::StorageStream& str) override;

    DOXYGEN_END_INTERNAL_BLOCK()

  protected:

#ifndef SWIG
    virtual agxPowerLine::PhysicalDimensionMultiBodyConstraintImplementation* createConstraint() override;

    /**
    Calculate the torque capacity based on the geometric and physical parameters of the friction plates.
    \note This will overwrite torque capacity set via setTorqueCapacity() method. The users are referred to the reference (Shaver R. Manual transmission clutch systems. Warrendale, Pennsylvania: SAE International, 1997.) for more information of the clutch parameters.
    \param maxForce - the maximum normal force exerted onto the frictional discs.
    \param mu - the Coulomb friction coefficient.
    \param radii - a 2D vector of the inner and outer radii of the frictional discs.
    */
    void calculateTorqueCapacity(agx::Real maxForce, agx::Real mu, agx::Real radii[2]);

    agx::RegularizationParameters::VariableType calculateComplianceAndDamping(const agx::Real timeStep, agx::Real& compliance, agx::Real& damping) override;

    bool preUpdate(agx::Real timeStep) override;

    bool addNotification(agxSDK::Simulation* simulation) override;

    bool postStore(agxStream::StorageStream& str) const override;

    bool postRestore(agxStream::StorageStream& str) override;
     
#endif

    virtual ~DryClutch();

    agx::Real m_torqueCapacity;
    bool      m_isEngaged;
    bool      m_isAutoLock;
    bool      m_isManual;
    agx::Real m_engageTimeConstant;
    agx::Real m_disengageTimeConstant;
    agx::Real m_minRelativeSlip;
    agx::Real m_fraction;
    bool      m_isLocked;

  };
  typedef agx::ref_ptr<DryClutch> DryClutchRef;

} // namespace agxDriveTrain
