#pragma once

#include <agx/Constraint.h>

namespace agx
{
  /**
  This constraint is meant as a container for a single 1DOF basic controller constraint without creating any elementary 
  constraints associated with the archetypal ordinary constraints.
  */
  class CALLABLE AGXPHYSICS_EXPORT SingleControllerConstraint1DOF : public Constraint1DOF
  {
    public:
      /**
      Create a 1DOF constraint with a single elementary constraints given an attachment frame on a rigid body and a fixed point (in world).
      \param rb1 - First rigid body (invalid if null)
      \param rb1AttachmentFrame - First rigid body attachment frame, the distance joint starts here (invalid if null)
      \param worldAttach - Attachment point in world, given i world coordinates. The distance joint ends here.
      \param controller - The BasicControllerConstraint that this constraint should contain, This controller is expected to have an angle but not an attachment pair.
      */
      SingleControllerConstraint1DOF( RigidBody* rb1, Frame* rb1AttachmentFrame, const Vec3& worldAttach, agx::BasicControllerConstraint* controller );

      /**
      Create a 1DOF constraint with a single elementary constraints given attachment frames and two rigid bodies.
      \param rb1 - First rigid body (invalid if null)
      \param rb1AttachmentFrame - First rigid body attachment frame, the distance joint starts here (invalid if null)
      \param rb2 - Second rigid body (invalid if null)
      \param rb2AttachmentFrame - Second rigid body attachment frame, the distance joint ends here (invalid if null)
      \param controller - The BasicControllerConstraint that this constraint should contain, This controller is expected to have an angle but not an attachment pair.
      */
      SingleControllerConstraint1DOF( RigidBody* rb1, Frame* rb1AttachmentFrame, RigidBody* rb2, Frame* rb2AttachmentFrame, agx::BasicControllerConstraint* controller );

      /**
      Create a 1DOF constraint with a single elementary constraints given two rigid bodies and local attachment points.
      \param rb1 - First rigid body (invalid if null)
      \param localAttachRB1 - First rigid body local attach point, the distance joint starts here
      \param rb2 - Second rigid body (invalid if null)
      \param localAttachRB2 - Second rigid body local attach point, the distance joint ends here
      \param controller - The BasicControllerConstraint that this constraint should contain, This controller is expected to have an angle but not an attachment pair.
      */
      SingleControllerConstraint1DOF( RigidBody* rb1, const Vec3& localAttachRB1, RigidBody* rb2, const Vec3& localAttachRB2, agx::BasicControllerConstraint* controller );

      /**
      Create a 1DOF constraint with a single elementary constraints given one rigid body, a local attach point and a world attach point.
      \param rb1 - First rigid body (invalid if null)
      \param localAttachRB1 - Local attach point on rb1, the distance joint starts here
      \param worldAttach - Attach point given in world coordinates, the distance joint ends here
      \param controller - The BasicControllerConstraint that this constraint should contain, This controller is expected to have an angle but not an attachment pair.
      */
      SingleControllerConstraint1DOF( RigidBody* rb1, const Vec3& localAttachRB1, const Vec3& worldAttach, agx::BasicControllerConstraint* controller );

      /**
      This is a 1D constraint, i.e., only index \p i = 0 is valid.
      \param i - is ignored because this constraint is 1D
      \return the regularization parameter
      */
      virtual RegularizationParameters* getRegularizationParameters( agx::UInt i ) override;
      virtual const RegularizationParameters* getRegularizationParameters( agx::UInt i ) const override;

      /**
      Set compliance for wrapped secondary constraint.
      */
      virtual void setCompliance( Real compliance, agx::Int index = -1 ) override;

      /**
      \return the compliance of the wrapped secondary constraint, \p index is ignored
      */
      virtual Real getCompliance( agx::UInt index ) const override;

      /**
      Set damping for the wrapped secondary constraint. 
      */
      virtual void setDamping( Real damping, agx::Int index = -1 ) override;

      /**
      \return the damping of the wrapped secondary constraint.
      */
      virtual Real getDamping( agx::UInt index ) const override;

      /**
      \return the number of DOF for this constraint, this will be 1.
      */
      virtual int getNumDOF() const override;

      AGXSTREAM_DECLARE_SERIALIZABLE(agx::SingleControllerConstraint1DOF);

    protected:
      SingleControllerConstraint1DOF();
      virtual ~SingleControllerConstraint1DOF();

      /**
      Internal method used for debug rendering
      */
      virtual void render(  agxRender::RenderManager *mgr, float scale   ) const override;

    private:
      class SingleControllerConstraint1DOFImplementation *m_implementation;
  };

  typedef ref_ptr<SingleControllerConstraint1DOF> SingleControllerConstraint1DOFRef;
} // namespace agx
