
#pragma once

#include <memory>
#include <unordered_set>
#include <unordered_map>
#include <openplx/Type.h>
#include <openplx/Node.h>
#include <openplx/TopologicalPath.h>
#include <openplx/ErrorReporter.h>

namespace openplx {

    class SymbolTree;

    /**
     * The nodes of a symbol tree. Each node stores a list of paths with a right hand side (m_valued_paths)
     * and nodes with a named type (m_typed_paths). The child nodes are mapped in a
     * std::unordered_map<std::string, std::shared_ptr<SymbolTreeNode>> where
     * the key is the next symbol in the path, ensuring paths in the child
     * nodes are prefixed (in terms of symbols) by the nodes in the parent.
     *
     * Terminology:
     *
     *   *growing* - Populate the tree by creating branches starting from a root model declaration
     *   *outer type* - The actual type of a symbol (depends on which model is being evaluated)
     */
    class SymbolTreeNode : public std::enable_shared_from_this<SymbolTreeNode> {
        public:
            static std::shared_ptr<SymbolTreeNode> create(std::weak_ptr<SymbolTree> symbol_tree, std::shared_ptr<ErrorReporter> error_reporter);

            TopoPathPtr insertValuedPathAndReportDuplicates(const TopoPathPtr& path);

            void invalidatePathsAndReportErrors(const TopoPathPtr& prefix, bool is_reference = false);
            void invalidateNestedDeclarations();

            std::shared_ptr<SymbolTreeNode> followPath(const TopoPathPtr& path, size_t max_index = 0);
            std::shared_ptr<SymbolTreeNode> followMemberAccess(const MemberAccessPtr& member_access, size_t offset);
            std::vector<TypePtr> followOuterTypes(const TopoPathPtr& path, size_t max_index = 0);

        private:
            SymbolTreeNode(std::weak_ptr<SymbolTree> symbol_tree, std::shared_ptr<ErrorReporter> error_reporter);

            /**
             * grow_outer_types and grow_max_paths traverses the tree in order of precedence by calling
             * grow_outer_types_from_members, grow_outer_types_from_types, grow_max_paths_from_members and grow_max_paths_from_types
             * recursively. The order of precedence is:
             *
             * 1. Types from local members have the highest precedence
             * 2. Types from referenced types have precedence in order of nesting (becomes only up until previous type)
             * 3. Types from parents have lowest precedence
             */
            void grow_outer_types(const TopoPathPtr& path, const ModelDeclPtr& until, std::unordered_set<TopoPathPtr>& loop_detector);
            void grow_outer_types_from_members(const TopoPathPtr& path, const std::vector<AnnotationPtr>& annotations, const std::vector<VarAssignPtr>& members);
            void grow_outer_types_from_types(const TopoPathPtr& path, const std::vector<VarAssignPtr>& members, std::unordered_set<TopoPathPtr>& loop_detector);
            void grow_max_paths(const TopoPathPtr& path, const ModelDeclPtr& until, std::unordered_set<TopoPathPtr>& loop_detector);
            void grow_max_paths_from_members(const TopoPathPtr& path, const std::vector<VarAssignPtr>& members, std::unordered_set<TopoPathPtr>& siblings, std::unordered_set<std::shared_ptr<SymbolTreeNode>>& duplicates);
            void grow_max_paths_from_types(const TopoPathPtr& path, const std::vector<VarAssignPtr>& members, std::unordered_set<TopoPathPtr>& loop_detector);
            void collectMaximalValuedPaths(std::vector<TopoPathPtr>& result);
            void collectMaximalValuedPathsForMemberAccess(const MemberAccessPtr& member_access, std::vector<TopoPathPtr>& result, size_t offset);
            void collectEmptyPaths(std::vector<TopoPathPtr>& result, const TopoPathPtr& prefix);
            void collectAnnotations(std::vector<TopoPathPtr>& result);

            std::shared_ptr<ErrorReporter> m_error_reporter;
            std::weak_ptr<SymbolTree> m_symbol_tree;
            VarAssignPtr m_outer_type;
            bool m_ref_tag = false;
            TopoPathPtr m_maximal_valued_path;
            std::unordered_map<std::string, std::shared_ptr<SymbolTreeNode>> m_branches;
            std::unordered_map<std::string, TopoPathPtr> m_annotations;
            friend class SymbolTree;
    };

    using SymTreeNodePtr = std::shared_ptr<SymbolTreeNode>;
}
