
#pragma once

#include <unordered_map>
#include <memory>
#include <openplx/TopologicalPath.h>
#include <openplx/Node.h>
#include <openplx/Type.h>
#include <openplx/Token.h>
#include <openplx/MemberAccess.h>
#include <openplx/SymbolTreeNode.h>
#include <openplx/ErrorReporter.h>

namespace openplx {
    /**
     * A data structure that store TopologicalPath instances for efficient lookup.
     * The root node branches out based on which model declaration or root
     * var declaration the path starts at. The following symbol tree nodes
     * branches out and stores nodes based on the path's symbol. It functions as
     * a sort of prefix tree where paths in child nodes are prefixed by the
     * paths in their parent (in terms of symbols).
     */
    class SymbolTree : public std::enable_shared_from_this<SymbolTree> {
        public:

            static std::shared_ptr<SymbolTree> fromRootNodes(const std::vector<NodePtr>& nodes, std::shared_ptr<ErrorReporter> error_reporter);

            TypePtr lookupTypeFromPath(const TopoPathPtr& path);
            TypePtr lookupTypeFromPathUntil(const TopoPathPtr& path, size_t max_index);
            TypePtr lookupTypeFromIdentifier(const TopoPathPtr& this_path, Token identifier);
            TypePtr lookupTypeFromMemberAccess(const TopoPathPtr& this_path, MemberAccessPtr member_access, size_t offset = 0);

            std::vector<TypePtr> lookupTypesFromPath(const TopoPathPtr& path);

            std::vector<TopoPathPtr> lookupEmptyTypes();
            std::vector<TopoPathPtr> lookupAnnotations();

            std::vector<TopoPathPtr> lookupMaximalValuedPathsFromFromIdentifier(const TopoPathPtr& this_path, Token identifier);
            std::vector<TopoPathPtr> lookupMaximalValuedPathsFromMemberAccess(const TopoPathPtr& this_path, MemberAccessPtr member_access, size_t offset);
            std::vector<TopoPathPtr> lookupMaximalValuedPathsFromPath(const TopoPathPtr& path);
            std::vector<TopoPathPtr> lookupMaximalValuedPaths();

            bool isPathReference(const TopoPathPtr& path);

            void invalidatePathsAndReportErrors();
        private:

            std::shared_ptr<SymbolTreeNode> followPath(const TopoPathPtr& path);

            SymbolTree(std::shared_ptr<ErrorReporter> error_reporter);

            std::unordered_map<NodePtr, SymTreeNodePtr> m_branches;
            std::shared_ptr<ErrorReporter> m_error_reporter;
    };

    using SymTreePtr = std::shared_ptr<SymbolTree>;
}
