
#pragma once

#include <openplx/NodeVisitor.h>
#include <openplx/TopologicalPath.h>
#include <openplx/SymbolTree.h>

namespace openplx {
    /**
     * Traverses the maximal valued paths from the symbol tree and updates each paths
     * edges list. Edges comes in two types:
     *
     * # Type 1 (references in right hand side):
     * Foo:
     *     x is Real: 3.0
     *     y is Real: x
     *
     * An edge is added from Foo.x to Foo.y since Foo.y references Foo.x
     * in the right hand side expression.
     *
     * # Type 2 (shadowing left hand sides):
     * Foo:
     *     x is Real: 2.0
     *     static fn create() -> Foo
     *
     * Bar:
     *     foo is Foo: Foo.create()
     *
     * Baz is Bar:
     *     foo.x: 3.0
     *
     * A link is added from Baz.Bar.foo to Baz.foo.x because the left hand side
     * of Baz.Bar.foo shadows the left hand side of Baz.foo.x.
     */
    class FindEdgesVisitor : private NodeVisitor {
        public:
            FindEdgesVisitor(SymTreePtr topo_tree, std::vector<TopoPathPtr> maximal_paths);

            void findEdgesType1(const TopoPathPtr& target);
            void findEdgesType2(const TopoPathPtr& target);

        private:
            void visitConstant(const ConstantPtr &constant) override;
            void visitCall(const CallPtr &call) override;
            void visitMemberAccess(const MemberAccessPtr &member_access) override;
            void visitArray(const ArrayPtr &array) override;
            void visitUnary(const UnaryPtr &unary) override;
            void visitBinaryOp(const BinaryOpPtr &binary_op) override;
            void visitIndexing(const IndexingPtr& indexing) override;
            void visitInitializer(const InitializerPtr& initializer) override;

            NodePtr nsLookupMemberAccess(const MemberAccessPtr& member_access);

            SymTreePtr m_tree;
            std::vector<TopoPathPtr> m_paths;
            DocPtr m_document;
            TopoPathPtr m_target;
            TopoPathPtr m_this_path;
    };
}
