#pragma once

#include <string>
#include <vector>
#include <memory>
#include <openplx/Types.h>
#include <openplx/Token.h>
#include <openplx/ErrorVisitor.h>

namespace openplx
{

class Error
{
public:

    static std::shared_ptr<Error> createOnToken(
        error_code_t error_code,
        const Token& token,
        const std::string& source_id);

    static std::shared_ptr<Error> createOnTokens(
        error_code_t error_code,
        const std::vector<Token>& tokens,
        const std::string& source_id);

    static std::shared_ptr<Error> create(
        error_code_t error_code,
        line_t from_line,
        col_t from_column,
        line_t to_line,
        line_t to_column,
        const std::string& source_id);

    static std::shared_ptr<Error> create(
        error_code_t error_code,
        line_t line,
        col_t column,
        const std::string& source_id);

    error_code_t getErrorCode() const;
    line_t getLine() const;
    col_t getColumn() const;
    std::string getSourceId() const;

    line_t getFromLine() const;
    col_t getFromColumn() const;
    line_t getToLine() const;
    col_t getToColumn() const;

    virtual std::string accept(ErrorVisitor& visitor) const;
    virtual ~Error() = default;

    template<typename T>
    static std::shared_ptr<T> as(std::shared_ptr<Error> error) {
        return std::dynamic_pointer_cast<T>(error);
    }

protected:
    Error(
        error_code_t error_code,
        line_t from_line,
        col_t from_column,
        line_t to_line,
        line_t to_column,
        const std::string& source_id);

    Error(
        error_code_t error_code,
        line_t line,
        col_t column,
        const std::string& source_id);

private:
    error_code_t m_error_code;
    line_t m_from_line;
    col_t m_from_column;
    line_t m_to_line;
    col_t m_to_column;
    std::string m_source_id;
};



class CycleFoundError : public Error
{
public:
    std::string getCyclicNodesAsString() const;

    std::string getModelName() const;
    static std::shared_ptr<CycleFoundError> create(const Token& token,
                                                    const std::string& source_id,
                                                    const std::string& model_name, const std::string& cyclic_nodes);
    std::string accept(ErrorVisitor& visitor) const override;

protected:
    CycleFoundError(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& model_name,
        const std::string& cyclic_nodes);

private:
    std::string m_model_name;
    std::string m_cyclic_nodes;
};



// A generic Error that has a String parameter
class StringParameterError : public Error
{
public:
    std::string getStringParameter() const;
    static std::shared_ptr<StringParameterError> createOnToken(
        error_code_t error_code,
        const Token& token,
        const std::string& source_id,
        const std::string& string_parameter);
    static std::shared_ptr<StringParameterError> createOnTokens(
        error_code_t error_code,
        const std::vector<Token>& tokens,
        const std::string& source_id,
        const std::string& string_parameter);
    static std::shared_ptr<StringParameterError> create(error_code_t error_code,
                                                        line_t line, col_t column,
                                                        const std::string& source_id,
                                                        const std::string& string_parameter);
    std::string accept(ErrorVisitor& visitor) const override;

protected:
    StringParameterError(
        error_code_t error_code,
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& string_parameter);

private:
    std::string m_string_parameter;
};



class MissingAssignmentError : public StringParameterError
{
public:
    static std::shared_ptr<MissingAssignmentError> create(line_t line, col_t column,
                                                            const std::string& source_id,
                                                            const std::string& var_decl_path);
    std::string getVarDeclarationPath() const;

protected:
    MissingAssignmentError(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& var_decl_path);
};



class CircularDependency : public StringParameterError
{
public:
    std::string getCycleAsString() const;
    static std::shared_ptr<CircularDependency> create(line_t line, col_t column,
                                                        const std::string& source_id,
                                                        const std::string& cyclic_nodes);

protected:
    CircularDependency(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& cyclic_nodes);
};



class ModelDeclarationNotFound : public StringParameterError
{
public:
    std::string getModelName() const;
    static std::shared_ptr<ModelDeclarationNotFound> create(line_t line, col_t column,
                                                            const std::string& source_id,
                                                            const std::string& model_name);

protected:
    ModelDeclarationNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& model_name);
};

class TypeNotFound : public StringParameterError
{
public:
    std::string getTypeSegments() const;
    static std::shared_ptr<TypeNotFound> create(const std::vector<Token>& type_segments, const std::string& source_id);

protected:
    TypeNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& type_segments);
};

class VariableNotFound : public StringParameterError
{
public:
    static std::shared_ptr<VariableNotFound> create(line_t line, col_t column,
                                                    const std::string& source_id,
                                                    const std::string& name);

protected:
    VariableNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);
};

class MemberNotFound : public StringParameterError
{
public:
    static std::shared_ptr<MemberNotFound> create(const Token& member_token, const std::string& source_id);

protected:
    MemberNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& member_identifier);
};

class MethodNotFound : public StringParameterError
{
public:
    static std::shared_ptr<MethodNotFound> create(const Token& name_token, const std::string& method_path, const std::string& source_id);

protected:
    MethodNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& method_path);
};

class SymbolAlreadyDefined : public StringParameterError
{
public:
    static std::shared_ptr<SymbolAlreadyDefined> create(line_t line, col_t column,
                                                    const std::string &source_id,
                                                    const std::string &name);

protected:
    SymbolAlreadyDefined(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);
};

class SymbolAlreadyAssigned : public StringParameterError
{
public:
    static std::shared_ptr<SymbolAlreadyAssigned> create(const Token& identifier,
                                                    const std::string& source_id);

protected:
    SymbolAlreadyAssigned(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);
};

class ImportedFileNotFound : public StringParameterError
{
public:
    static std::shared_ptr<ImportedFileNotFound> create(line_t line, col_t column,
                                                    const std::string& source_id,
                                                    const std::string& name);

protected:
    ImportedFileNotFound(
        line_t from_line, col_t from_column,
        line_t to_line, col_t to_column,
        const std::string& source_id,
        const std::string& name);
};

class NoPluginForImportedFile : public StringParameterError
{
    public:
        static std::shared_ptr<NoPluginForImportedFile> create(line_t line, col_t column,
                                                               const std::string& source_id,
                                                               const std::string& name);

    protected:
        NoPluginForImportedFile(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& source_id,
            const std::string& name);
};

class FileNotFound : public StringParameterError
{
    public:
        static std::shared_ptr<FileNotFound> create(line_t line, col_t column,
                                                    const std::string &source_id,
                                                    const std::string &name);

    protected:
        FileNotFound(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& source_id,
            const std::string& name);
};

class ExpectedTrait : public StringParameterError
{
    public:
        static std::shared_ptr<ExpectedTrait> create(const std::vector<Token>& segments,
                                                     const std::string& source_id);

    protected:
        ExpectedTrait(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& source_id,
            const std::string& trait);
};

class ModelNotInitializable : public StringParameterError
{
    public:
        static std::shared_ptr<ModelNotInitializable> create(
            line_t line, col_t column,
            const std::string& model_name,
            const std::string& source_id);

    protected:
        ModelNotInitializable(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& source_id,
            const std::string& model_name);
};

class DependencyNotSatisfied : public StringParameterError
{
    public:
        static std::shared_ptr<DependencyNotSatisfied> create(
            line_t line, col_t column,
            const std::string& dependency,
            const std::string& source_id);

    protected:
        DependencyNotSatisfied(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& dependency,
            const std::string& source_id);
};

class ReferencesAreReadOnly : public StringParameterError
{
    public:
        static std::shared_ptr<ReferencesAreReadOnly> create(
            line_t line, col_t column,
            const std::string& reference,
            const std::string& source_id);

    protected:
        ReferencesAreReadOnly(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& reference,
            const std::string& source_id);
};

class ReferencesMustBeModels : public StringParameterError
{
    public:
        static std::shared_ptr<ReferencesMustBeModels> create(
                Token first_token, Token reference_token,
                const std::string& reference,
                const std::string& source_id);

    protected:
        ReferencesMustBeModels(
                line_t from_line, col_t from_column,
                line_t to_line, col_t to_column,
                const std::string& reference,
                const std::string& source_id);
};

class ReferencesCannotBecome : public StringParameterError
{
    public:
        static std::shared_ptr<ReferencesCannotBecome> create(
                Token first_token, Token reference_token,
                const std::string& reference,
                const std::string& source_id);

    protected:
        ReferencesCannotBecome(
                line_t from_line, col_t from_column,
                line_t to_line, col_t to_column,
                const std::string& reference,
                const std::string& source_id);
};


class TypeMismatch : public StringParameterError
{
    public:
        static std::shared_ptr<TypeMismatch> create(
                Token first_token, Token last_token,
                const std::string& lhs_type,
                const std::string& rhs_type,
                const std::string& source_id);

    protected:
        TypeMismatch(
            line_t from_line, col_t from_column,
            line_t to_line, col_t to_column,
            const std::string& string_param,
            const std::string& source_id);
};

using Errors = std::vector<std::shared_ptr<Error>>;
}
